/*
 * @(#)PaddleWar.java
 *
 * Last change: 2001-05-03
 *
 * Author: Andreas Dangel <a.dangel@astrowalk.de>
 *
 * A simple ping-pong game
 */

import java.applet.Applet;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.util.Date;


public class PaddleWar extends Applet implements Runnable {

    static final String VERSION = "ver.2001-05-03";
    static final boolean BORDER = true;
    static final int INFOBAR_HEIGHT = 50;
    static final int BALL_DIAMETER = 5;
    static final int PADDLE_WIDTH = 10;
    static final int PADDLE_HEIGHT = 5;
    static final int PLAYER_MOVEMENT = 5;
    static final int COMPUTER_MOVEMENT = 1;
    static final int LOOP_THREAD_DELAY = 20;

    Image offscreenImage = null;
    Graphics offscreen = null;
    Dimension dimension = null;

    Font font = null;
    FontMetrics fm = null;

    Thread loopThread = null;
    Thread timer = null;

    int time = 0;
    int timeParameter = 0;

    boolean playing = false;
    boolean paused = false;
    boolean left = false;
    boolean right = false;

    Point player = null;
    Point computer = null;
    Point ball = null;

    int ballDirectionX = 0;
    int ballDirectionY = 0;
    int ballMovementX = 0;

    int playerScore = 0;
    int computerScore = 0;

    public String getAppletInfo() {
        return "PaddleWar " + VERSION + "\n"
             + "(c) 2001 Andreas Dangel <a.dangel@astrowalk.de>";
    }

    public String[][] getParamterInfo() {
        String[][] parameters = {
            {"time", "int", "the duration of one game in seconds"}
        };
        return parameters;
    }


    public void init() {
        dimension = size();

        offscreenImage = createImage(dimension.width, dimension.height);
        offscreen = offscreenImage.getGraphics();

        font = new Font("Helvetica", Font.BOLD, 14);
        fm = getFontMetrics(font);

        player = new Point();
        computer = new Point();
        ball = new Point();

        String timeStr = getParameter("time");
        if (timeStr == null) {
            timeStr = "90";
        }
        try {
            timeParameter = Integer.parseInt(timeStr);
        } catch (NumberFormatException e) {
            timeParameter = 90;
        }

        newGame();
    }

    public void start() {
        if (loopThread == null) {
            loopThread = new Thread(this);
            loopThread.start();
        }

        if (timer == null) {
            timer = new Thread(this);
            timer.start();
        }
    }

    public void stop() {
        if (loopThread != null) {
            loopThread = null;
        }
        if (timer != null) {
            timer = null;
        }
    }

    public void destroy() {
        offscreen.dispose();
    }

    public void run() {
        Thread thisThread = Thread.currentThread();

        while (loopThread == thisThread) {
            if (playing && !paused) {
                updatePlayer();
                updateComputer();
                updateBall();
                checkCollisions();
            }
            repaint();
            try {
                Thread.sleep(LOOP_THREAD_DELAY);
            } catch (InterruptedException e) { }
        }

        while (timer == thisThread) {
            if (playing && !paused) {
                time--;
            }
            repaint();
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) { }
        }
    }

    public void update(Graphics screen) {
        paint(screen);
    }

    public void paint(Graphics screen) {
        int x, y;
        String s;

        offscreen.setColor(Color.white);
        offscreen.fillRect(0, 0, dimension.width, dimension.height);

        offscreen.setColor(Color.yellow);
        offscreen.fillRect(0, 0, dimension.width, INFOBAR_HEIGHT);

        offscreen.setColor(Color.black);
        offscreen.fillRect(player.x, player.y, PADDLE_WIDTH, PADDLE_HEIGHT);
        offscreen.fillRect(computer.x, computer.y, PADDLE_WIDTH,
                           PADDLE_HEIGHT);
        offscreen.fillOval(ball.x, ball.y, BALL_DIAMETER, BALL_DIAMETER);


        offscreen.setFont(font);

        y = INFOBAR_HEIGHT - 3;

        x = 3;
        s = "comp: " + computerScore;
        offscreen.drawString(s, x, y);

        s = "you: " + playerScore;
        x = dimension.width - fm.stringWidth(s) - 3;
        offscreen.drawString(s, x, y);

        s = "time: " + time + "s";
        x = (int) dimension.width / 2 - (int) fm.stringWidth(s) / 2;
        offscreen.drawString(s, x, y);


        y = fm.getHeight() + 3;
        s = VERSION;
        x = dimension.width - fm.stringWidth(s) - 3;
        offscreen.drawString(s, x, y);




        if (playing && paused) {
            s = "paused... press 'p' to continue";
            y = (int) dimension.height / 2 - (int) fm.getHeight() / 2;
            x = (int) dimension.width / 2 - (int) fm.stringWidth(s) / 2;
            offscreen.drawString(s, x, y);
        }

        if (!playing) {
            s = "press 's' to start...";
            y = (int) dimension.height / 2 - (int) fm.getHeight() / 2;
            x = (int) dimension.width / 2 - (int) fm.stringWidth(s) / 2;
            offscreen.drawString(s, x, y);
        }

        if (BORDER) {
            offscreen.drawRect(0, 0, dimension.width - 1,
                               dimension.height - 1);
            offscreen.drawLine(0, INFOBAR_HEIGHT, dimension.width - 1,
                               INFOBAR_HEIGHT);
        }

        screen.drawImage(offscreenImage, 0, 0, this);
    }

    public boolean keyDown(Event evt, int key) {
        if (key == 's' && !playing) {
            initGame();
        } else if (key == 'p' && playing) {
            paused = !paused;
        } else if (key == Event.LEFT && playing) {
            left = true;
        } else if (key == Event.RIGHT && playing) {
            right = true;
        }
        return true;
    }

    public boolean keyUp(Event evt, int key) {
        if (key == Event.LEFT) {
            left = false;
        } else if (key == Event.RIGHT) {
            right = false;
        }
        return true;
    }

    public void initGame() {
        time = timeParameter;

        playerScore = 0;
        computerScore = 0;

        player.x = 0;
        player.y = dimension.height - 10 - PADDLE_HEIGHT;
        computer.x = 0;
        computer.y = INFOBAR_HEIGHT + 10;

        ball.x = (int) dimension.width  / 2 + (int) BALL_DIAMETER / 2;
        ball.y = (int) (dimension.height - INFOBAR_HEIGHT) / 2
                - (int) BALL_DIAMETER / 2;
        ball.y += INFOBAR_HEIGHT;

        ballDirectionX = 1;
        ballDirectionY = 1;
        ballMovementX = 1;

        paused = false;
        playing = true;
    }

    public void newGame() {
        initGame();

        playing = false;
        paused = false;
    }

    public void updatePlayer() {
        if (left) {
            player.x -= PLAYER_MOVEMENT;
        }
        if (right) {
            player.x += PLAYER_MOVEMENT;
        }

        if (player.x < 0) {
            player.x = 0;
        } else if (player.x > (dimension.width - PADDLE_WIDTH)) {
            player.x = dimension.width - PADDLE_WIDTH;
        }
    }

    public void updateComputer() {
        int distance;
        int direction;

        distance = (ball.x + (int) BALL_DIAMETER / 2)
                 - (computer.x + (int) PADDLE_WIDTH / 2);

        if (distance > 0) {
            direction = +1;
        } else if (distance < 0) {
            direction = -1;
        } else {
            direction = 0;
        }

        distance = Math.abs(distance);
        if (distance >= COMPUTER_MOVEMENT) {
            computer.x += COMPUTER_MOVEMENT * direction;

            if (computer.x < 0) {
                computer.x = 0;
            } else if (computer.x > (dimension.width - PADDLE_WIDTH)) {
                computer.x = dimension.width - PADDLE_WIDTH;
            }
        }
    }

    public void updateBall() {
        ball.y += ballDirectionY * 1;
        ball.x += ballDirectionX * ballMovementX;
    }

    public void resetBall() {
        ball.x = (int) dimension.width  / 2 + (int) BALL_DIAMETER / 2;
        ball.y = (int) (dimension.height - INFOBAR_HEIGHT) / 2
                - (int) BALL_DIAMETER / 2;
        ball.y += INFOBAR_HEIGHT;
        repaint();

        stop();
        try {
            Thread.sleep(250);
        } catch (InterruptedException e) { }
        start();
    }

    public void checkCollisions() {
        if (time <= 0) {
            playing = false;
        }

        if (ball.x < 0) {
            ballDirectionX *= -1;
        } else if (ball.x > (dimension.width - BALL_DIAMETER)) {
            ballDirectionX *= -1;
        }

        if (ball.y < INFOBAR_HEIGHT) {
            playerScore += 1;
            resetBall();
        } else if (ball.y > (dimension.height - BALL_DIAMETER)) {
            computerScore += 1;
            resetBall();
        }

        int distance;

        if ((ball.y + BALL_DIAMETER) == player.y) {
            distance = Math.abs(ball.x - player.x);
            if (distance <= PADDLE_WIDTH) {
                ballDirectionY *= -1;
                if (left && (ballDirectionX == -1)) {
                    ballMovementX += 1;
                } else if (left && (ballDirectionX == +1)) {
                    ballMovementX -= 1;
                } else if (right && (ballDirectionX == -1)) {
                    ballMovementX -= 1;
                } else if (right && (ballDirectionX == +1)) {
                    ballMovementX += 1;
                }
                if (ballMovementX > COMPUTER_MOVEMENT) {
                    ballMovementX = COMPUTER_MOVEMENT;
                }
            }
        }

        if (ball.y == (computer.y + PADDLE_HEIGHT)) {
            distance = Math.abs(ball.x - computer.x);
            if (distance <= PADDLE_WIDTH) {
                ballDirectionY *= -1;
            }
        }
    }
}