/*
 * Huffman
 * 
 * Andreas Dangel <it04109@informatik.ba-stuttgart.de>
 * 15.07.2005
 * 
 * In der Klasse Huffman ist die main-Methode implementiert,
 * die die Kompression / Dekompression verwaltet.
 * 
 * Es wird mindestens Java 1.4 bentigt!!! (z.B. StringBuffer.indexOf(String, int))
 */

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.FileNotFoundException;

public class Huffman {

	/*
	 * Konstanten, die die ausgewhlte Funktion
	 * des Programms darstellen.
	 */
    public static final int ACTION_NOTHING = 0;
    public static final int ACTION_COMPRESS = 1;
    public static final int ACTION_DECOMPRESS = 2;

    public static void main(String[] args) {

    	/* zu wenig Argumente... */
        if (args.length < 1) {
            show_help();
        }


        /* 
         * Optionen interpretieren
         */
        
        // Anfangswerte der Optionen
        boolean option_show_codetable = false;
        boolean option_show_tree = false;
        int action = ACTION_NOTHING;
        String filename = null;
        String filename_in = null;
        String filename_out = null;
        File file_in = null;
        File file_out = null;

        /*
         * Jetzt alle Argumente der Reihe nach durchgehen
         */
        for (int args_index = 0; args_index < args.length; args_index++) {
        	// Ein Parameter auswhlen
            String o = args[args_index];

            // Wenn der Parameter nur ein Zeichen lang ist
            // und bis jetzt noch keine Aktion gewhlt wurde,
            // dann ist das wahrscheinlich die gewnschte Aktion.
            if (o.length() == 1 && action == ACTION_NOTHING) {
                if (o.charAt(0) == 'c') {
                    action = ACTION_COMPRESS;
                } else if (o.charAt(0) == 'd') {
                    action = ACTION_DECOMPRESS;
                } else {
                    show_error("invalid action");
                }
            // Erstes Zeichen des Parameters ist "-" und es wurde
            // noch keine Aktion ausgewhlt -> wahrscheinlich ist
            // eine Option angegeben worden ("-x" oder "-y")
            } else if (o.charAt(0) == '-' && action == ACTION_NOTHING) {
                if (o.length() == 2) {
                    if (o.charAt(1) == 'x') {
                        option_show_codetable = true;
                    } else if (o.charAt(1) == 'y') {
                        option_show_tree = true;
                    } else {
                        show_error("invalid option");
                    }
                } else {
                    show_error("invalid option");
                }
            // Wenn bereits eine Aktion gewhlt wurde und noch kein
            // Dateiname angegeben wurde, dann ist der aktuelle
            // Parameter der Dateiname.
            } else if (filename == null && action != ACTION_NOTHING) {
                filename = o;
            } else {
                show_error("too many parameters!");
            }
        }


        // Zustzliche berprfung, falls keine Aktion angegeben wurde.
        if (action == ACTION_NOTHING) {
            show_error("shall I compress or decompress?");
        }


        /*
         * Eingabe und Ausgabestrme
         * Es wird der Eingabestrom komprimiert/dekomprimiert
         * Das Ergebnis wird im Ausgabestream ausgegeben
         * 
         * Hier gleich Standardwerte:
         * System.in und System.out
         */
        InputStream input = System.in;
        OutputStream output = System.out;


        /*
         * Den zweiten Dateinamen fr die Ausgabe
         * beschaffen,
         * nur wenn ein Dateiname angegeben wurde
         * (sonst System.in/System.out) 
         */
        if (filename != null) {
            // beim Komprimieren die Endung ".huffman" hinzufgen
            if (action == ACTION_COMPRESS) {
                filename_in = filename;
                filename_out = filename_in + ".huffman";
            // beim Dekomprimieren, die Endung ".huffman" entfernen
            // und eventuell Fehlermeldung ausgeben, falls die
            // erste Datei nicht auf ".huffman" endet.
            } else if (action == ACTION_DECOMPRESS) {
                filename_in = filename;
                if (filename_in.endsWith(".huffman")) {
                    filename_out = filename_in.substring(0, filename_in.length()-".huffman".length());
                } else {
                    show_error("the input filename doesn't end with \".huffman\". is it really compressed with this program?");
                }
            }

            
            // die Dateiobjekte erstellen
            file_in = new File(filename_in);
            file_out = new File(filename_out);
            
            // Und berprfen, ob die Eingabe-Datei berhaupt existiert.
            if (!file_in.exists()) {
                show_error("File \"" + filename_in + "\" doesn't exist");
            }
            // Die Ausgabedatei darf nicht existieren
            if (file_out.exists()) {
            	show_error("File \"" + filename_out + "\" already exists");
            }

            // Beide Dateien ffnen
            try {
                input = new FileInputStream(file_in);
                output = new FileOutputStream(file_out);
            } catch (FileNotFoundException e) {
            	show_error("file not found!");
            }
        }

        /*
         * BufferedReader und BufferedWriter erstellen
         */
        BufferedInputStream buf_input = new BufferedInputStream(input);
        BufferedOutputStream buf_output = new BufferedOutputStream(output);
        
        
        /*
         * Die eigentliche Arbeit geschieht hier....
         */
        try {
        	// speichert ein einzelnes Zeichen
            int c;
            
            // speichert die gesamte Datei/die gesamten Daten
            StringBuffer data = new StringBuffer();
            
            // der Huffman-Tree, der zum komprimieren/dekomprimieren
            // verwendet wird.
            HuffmanTree tree = new HuffmanTree();

            // Die Datei bzw. stdin einlesen.
            while ((c = buf_input.read()) != -1) {
            	// das Zeichen der Datensammlung hinzufgen
                data.append((char) c);
                
                // dem Baum das Zeichen bergeben, damit die
                // Hufigkeit ausgezhlt werden kann
                tree.addData(c);
            }

            /*
             * KOMPRIMIEREN 
             */
            if (action == ACTION_COMPRESS) {
                // Nach dem Einlesen den Baum erstellen lassen
                tree.buildTree();
                
                // Eventuell die Code-Tabelle oder den Baum
                // ausgeben lassen 
                if (option_show_codetable == true) {
                    tree.printCodetable(System.err);
                } else if (option_show_tree == true) {
                    tree.printBaum(System.err);
                }

                // die Code-Tabelle erstellen lassen
                StringBuffer codetable = tree.createRawCodetable();
                // Code-Tabelle Zeichen fr Zeichen ausgeben
                for(int i = 0; i < codetable.length(); i++) {
                	// die Methode write wandelt das Zeichen
                	// - im Gegensatz zu print - nicht in den
                	// Zeichensatz des Betriebssystems um.
                    buf_output.write((int) codetable.charAt(i));
                }
                
                // Jetzt die eigentliche Kompression
                StringBuffer encoded = tree.encode(data);
                // Auch wieder Zeichenweise ausgeben
                for(int i = 0; i < encoded.length(); i++) {
                    buf_output.write((int) encoded.charAt(i));
                }
                // die Zeichen, die noch im Buffer sind, ausgeben.
                buf_output.flush();

                // Status-Informationen ausgeben
                if (file_in != null) {
                	show_status("\"" + file_in + "\" to \"" + file_out + "\" compressed...");
                }
                // Kompressions-Rate berechnen und ausgeben
                String ratio_string = "compression ratio: ";
                float ratio = (codetable.length() + encoded.length())/(float) data.length();
                ratio *= 100f;
                ratio = 100f - ratio;
                ratio = ((int) ratio * 10f) / 10f;
                ratio_string += ratio + "%";
                show_status(ratio_string);
            }
            else if (action == ACTION_DECOMPRESS) {
                /*
                 * DEKOMPRIMIEREN 
                 */

            	// die Codetabelle am Anfang der Datei/des InputStreams
            	// einlesen lassen
            	// index ist die aktuelle Position in data.
                int index = tree.readRawCodetable(data);

                if (option_show_codetable == true) {
                    tree.printCodetable(System.err);
                } else if (option_show_tree == true) {
                    tree.printBaum(System.err);
                }

                // dekomprimieren lassen
                StringBuffer decoded = tree.decode(data, index);
                
                // Zeichenweise ausgeben (mit write, s.o.)
                for(int i = 0; i < decoded.length(); i++) {
                    buf_output.write((int) decoded.charAt(i));
                }
                buf_output.flush();
                
                // Statusausgabe mit den Dateinamen, falls eine Datei
                // dekomprimiert wurde.
                if (file_in != null) {
                	show_status("\"" + file_in + "\" to \"" + file_out + "\" decompressed...");
                }
            }
            
            /*
             * Ein- und Ausgabestrme schlieen
             */
            buf_input.close();
            buf_output.close();
            input.close();
            output.close();

        } catch (IOException e) {
        	// Fehlermeldung ausgeben
            show_error(e.toString());
        }


    }

    /*
     * Gibt eine Statusmeldung aus.
     */
    private static void show_status(String msg) {
    	System.err.println(msg);
    }
    
    /*
     * Gibt eine Fehlermeldung aus und danach
     * die Aufruf-Syntax des Programms, wobei dann
     * die Ausfhrung des Programms beendet wird.
     */
    private static void show_error(String msg) {
    	System.err.println("error:\n" + msg + "\n");
    	show_help();
    }
    
    /*
     * Gibt die Aufruf-Syntax des Programms aus
     * und beendet die Ausfhrung.
     */
    private static void show_help() {
        System.err.println("Syntax: java -jar Huffman.jar [-x|-y] {c|d} [filename]\n");
        System.err.println("   -x = print codetable");
        System.err.println("   -y = print tree");
        System.err.println("    c = compress");
        System.err.println("    d = decompress");
        System.err.println("    if no filename ist given, then stdin to stdout is compressed/decompressed");
        System.exit(1);
    }


}
