/*
 * HuffmanTree
 * 
 * Andreas Dangel <it04109@informatik.ba-stuttgart.de>
 * 15.07.2005
 * 
 * Diese Klasse reprsentiert einen Huffman-Baum. Mit der
 * Methode addData(char) werden die Hufigkeiten ausgezhlt,
 * mit buildTree() wird ein Baum aufgebaut. Die einzelnen
 * Baum-Element, d.h. die Knoten, sind Objekte der Klasse Node.
 * Auerdem gibt es Methoden zum Komprimieren (encode()) und zum
 * Dekomprimieren (decode()).
 * 
 */

import java.util.Vector;
import java.io.PrintStream;



public class HuffmanTree {

    /*
     * speichert die Hufigkeitsverteilung der Zeichen
     */
    private long characters[] = new long[256];
    
    /*
     * speichert die Anzahl der Zeichen vor dem Komprimieren
     * wird vor allem beim Dekomprimieren bentigt.
     */
    private long characterscount = 0;
    
    /*
     * speichert die Code-Tabelle als Strings (z.B. "010100")
     */
    private String codetable[] = new String[256];
    
    /*
     * speichert die Wurzel des Baumes
     */
    private Node root = null;
    
    
    
    
    /*
     * addData(int c):
     * wird benutzt, um den Baum aufzubauen
     * diese Funktion zhlt die Hufigkeit der Zeichen aus
     */
    public void addData(int c) {
        if (c >= 0 && c <= 255) {
            characters[c]++;
        }
        characterscount++;
    }
    
    /*
     * buildTree():
     * Nachdem mit addData() die Hufigkeiten ausgezhlt wurden,
     * kann mit buildTree() der Baum erstellt werden
     */
    public void buildTree() {

        /*
         * Baum aufbauen
         * Schritt 1:
         * Die Bltter erstellen 
         */
        
        // Dieser Vector speichert die Bltter
        Vector blaetter = new Vector();
        // Fr die Code-Tabelle eine Sicherung anlegen
        // damit spter ganz einfach auf die Zeichen
        // zugegriffen werden kann.
        Vector blaetter2 = new Vector();
        for(int i = 0; i < 256; i++) {
            // kommt das Zeichen berhaupt vor?
            if (characters[i] > 0) {
                // Neues Blatt/Knoten erstellen
                Node n = new Node();
                n.setData(i);
                n.setCount(characters[i]);
                // und in beide Vectoren speichern
                blaetter.add(n);
                blaetter2.add(n);
            }
        }
        
        /*
         * Das ganze jetzt solange wiederholen, bis nur noch
         * ein Blatt brig ist.
         * Dieses eine Blatt ist dann die Wurzel des Baumes
         */
        while(blaetter.size() > 1) {
            
            /* Schritt 2:
             * Die zwei seltensten Zeichen suchen
             * Die entsprechenden Bltter mssen dann aus dem Vector
             * herausgenommen werden und extra gespeichert werden (min1, min2)
             */
            Node min1, min2;
            min1 = (Node) blaetter.remove(0);
            for(int i = 0; i < blaetter.size(); i++) {
                Node temp = (Node) blaetter.get(i);
                // kleiner?
                if (temp.getCount() < min1.getCount()) {
                    // dann tauschen
                    blaetter.remove(i);
                    blaetter.add(min1);
                    min1 = temp;
                }
            }
            
            // Das ganze fr min2 nochmal
            min2 = (Node) blaetter.remove(0);
            for(int i = 0; i < blaetter.size(); i++) {
                Node temp = (Node) blaetter.get(i);
                //kleiner?
                if (temp.getCount() < min2.getCount()) {
                    // tauschen
                    blaetter.remove(i);
                    blaetter.add(min2);
                    min2 = temp;
                }
            }
            
            
            /*
             * Schritt 3:
             * Aus den zwei Blttern ein neuer Knoten machen und verknpfen
             * Dieser neue Knoten dann wieder dem Vector hinzufgen
             */
            Node n = new Node();
            n.setCount(min1.getCount() + min2.getCount());
            n.setLeftChild(min1);
            n.setRightChild(min2);
            min1.setParent(n);
            min2.setParent(n);

            blaetter.add(n);

        }


        /*
         * 4. Schritt:
         * Der brige Knoten ist die Wurzel des Baumes
         */
        root = (Node) blaetter.remove(0);

        /*
         * 5. Schritt
         * Die Code-Tabelle erstellen
         */
        // den Sicherungsvektor durchgehen
        for(int i = 0; i < blaetter2.size(); i++) {
            Node n = (Node) blaetter2.get(i);
            String code = "";
            int data = n.getData();

            // den Baum von den Blttern zur Wurzel
            // durchgehen ("von unten nach oben")
            while (n.getParent() != null) {
                Node n2 = n.getParent();
                if (n2.getLeftChild() == n) {
                    code = "0" + code;
                } else {
                    code = "1" + code;
                }
                n = n2;
            }

            // Falls wir nur ein zeichen hatten und die Wurzel
            // gleich das einzige Blatt ist...
            if (code.length() == 0) code = "0";

            codetable[data] = code;
        }
        
        /*
         * Jetzt ist der Baum fertig erstellt.
         * Er kann intern ber die private Variable
         * "root" angesprochen werden.
         * 
         * Auerdem wurde die Codetabelle erstellt, die intern
         * ber das Array "codetable" zur Verfgung steht.
         */
    }

    /*
     * printCodetable(PrintStream):
     * Alle Zeichen der Codetabelle mit ihrem Code
     * tabellarisch ausgeben.
     * Beispiel:
     * 33 [!]: 00011001
     */
    public void printCodetable(PrintStream o) {
        for(int i = 0; i < 256; i++) {
        	// nur Ausgeben, wenn ein Code existiert;
        	// wenn nicht, kam das Zeichen auch nicht vor
            if (codetable[i] != null) {
                o.print(i + " [");
                
                // das Zeichen selber ausgeben, aber nur,
                // wenn es kein Steuerzeichen ist. Das erste
                // Zeichen das direkt ausgegeben wird, hat den
                // ASCII-Code 32 (Leerzeichen).
                if (i > 31) o.print((char) i);
                // Ansonsten fr Steuerzeichen ein Punkt ausgeben
                else o.print(".");

                o.println("]: " + codetable[i]);
            }
        }
    }


    /*
     * createRawCodetable():
     * Erstellt die Code-Tabelle, wie sie spter in der
     * komprimierten Datei am Anfang steht.
     * Rckgabe: als StringBuffer
     */
    public StringBuffer createRawCodetable() {
        StringBuffer ct = new StringBuffer();

        // als ersten Integer die Gre der zu
        // komprimierenden Datei schreiben
        ct.append(RawInteger.getRawData((int) characterscount));

        // Codetabelle durchgehen
        for(int i = 0; i < 256; i++) {
            if (codetable[i] != null) {

                // Der Binrcode muss so gespeichert werden, dass
                // fhrende Nullen erhalten bleiben
                // Dazu wird eine fhrende 1 hinzugefgt, die spter
                // wieder entfernt wird.
                String code = new String(codetable[i]);
                code = "1" + code;

                // jetzt auf ein Vielfaches von 8 Bit mit "0" auffllen
                // (diese fhrende Nullen sind unwichtig und werden spter
                // inklusive der "1" von oben ignoriert)
                while (code.length() % 8 != 0) {
                    code = "0" + code;
                }

                // Anzahl der Bytes, in der der Code gespeichert ist
                int count_bytes = ((code.length()-1) / 8) + 1;
                
                // Diese Anzahl schreiben
                ct.append(RawInteger.getRawData(count_bytes));

                // Und dann zunchst das Zeichen selber schreiben
                ct.append((char) i);

                // Und jetzt der Code, Byte fr Byte
                for(int j = 0; j < count_bytes; j++) {
                    int b = Integer.parseInt(code.substring(j*8, (j+1)*8), 2);
                    ct.append((char) b);
                }

            }
        }

        // Ende-Markierung der Code-Tabelle mit einem
        // Integer-Wert von 0
        ct.append(RawInteger.getRawData(0));

        return ct;
    }


    /*
     * readRawCodetable(StringBuffer):
     * Liest die Code-Tabelle aus dem StringBuffer aus.
     * Gibt den Index zurck, ab dem die eigentlichen Daten
     * beginnen.
     * Auerdem wird gleich der Huffman-Baum erstellt
     */
    public int readRawCodetable(StringBuffer data) {
        // Codetabelle zuerst lschen
        for (int k = 0; k < 256; k++) codetable[k] = null;

        // das Array speichert jeweils 4 Bytes, die einen
        // Integer reprsentieren
        char[] rawint = new char[4];
        // der dazugehrige Integer
        int i;

        // speichert die Position im bergebenenen
        // StringBuffer data. Wird am Ende zurckgegeben.
        int index = 0;

        // Den ersten Integer einlesen (4 Zeichen)
        rawint[0] = data.charAt(index++);
        rawint[1] = data.charAt(index++);
        rawint[2] = data.charAt(index++);
        rawint[3] = data.charAt(index++);
        i = RawInteger.parseRawData(rawint);

        // der erste Integer ist die Gre der dekomprimierten Datei
        characterscount = i;

        
        // den nchsten Integer einlesen (4 Zeichen)
        rawint[0] = data.charAt(index++);
        rawint[1] = data.charAt(index++);
        rawint[2] = data.charAt(index++);
        rawint[3] = data.charAt(index++);
        i = RawInteger.parseRawData(rawint);

        // Wenn i == 0, dann ist die Ende-Markierung der
        // Codetabelle erreicht.
        while (i != 0) {
            // Ansonsten speichert i, wieviele Bytes lang
            // das Codewort des folgenden Zeichens ist.
        	
            // Also zuerst das Zeichen
            int c = (int) data.charAt(index++);

            // und jetzt das Codewort (mit fhrenden "0" und einer
            // fhrenden "1")
            String rawString = new String();
            for(int j = 0; j < i; j++) {
                int c2 = (int) data.charAt(index++);
            	// Bei ungltigen Daten kommt es vor,
            	// dass die Codetabelle nicht aufhrt,
            	// d.h. index >= data.length()
            	if (index >= data.length()) {
            		System.err.println("error: invalid file format!");
            		return index;
            	}
		
                // die fhrenden Nullen werden von toBinaryString(int) weggelassen
                String rs = Integer.toBinaryString(c2);
                // fhrende wieder Null hinzufgen
                while (rs.length() % 8 != 0) rs = "0" + rs;

                rawString += rs;
            }

            // der Code enthlt jetzt noch eventuell fhrende Nullen und die
            // fhrende "1"; das alles muss entfernt werden
            rawString = rawString.substring(rawString.indexOf('1') + 1);

            // jetzt der Codetabelle hinzufgen
            codetable[c] = new String(rawString);

            // nchsten Integer einlesen
            rawint[0] = data.charAt(index++);
            rawint[1] = data.charAt(index++);
            rawint[2] = data.charAt(index++);
            rawint[3] = data.charAt(index++);
            i = RawInteger.parseRawData(rawint);
        }


        // Jetzt den Baum erstellen
        root = new Node();
	// dazu die Codetabelle durchgehen
        for(i = 0; i < 256; i++) {
            if (codetable[i] != null) {
                Node n = root;

                String code = codetable[i];
		// die einzelnen Bits des Codewortes durchgehen
                for (int j = 0; j < code.length(); j++) {
                    if (code.charAt(j) == '0') {
			// eventuell existiert das Kind noch nicht
                        if (n.getLeftChild() == null) {
                            n.setLeftChild(new Node());
                        }
			// eine Stufe tiefer gehen im Baum
                        n = n.getLeftChild();
                    } else {
			// eventuell existiert das Kind noch nicht    
                        if (n.getRightChild() == null) {
                            n.setRightChild(new Node());
                        }
			// eine Stufe tiefer gehen im Baum
                        n = n.getRightChild();
                    }
                }

		// n zeigt jetzt auf den letzten Knoten
		// dies ist das Blatt. Jetzt muss noch das Zeichen
		// gespeichert werden, das dieses Blatt reprsentiert.
                n.setData(i);
            }
        }


	// aktuelle Position im StringBuffer data zurckgeben
        return index;
    }


    /*
     * printBaum(PrintStream):
     * druckt den Baum in einen beliebigen Stream aus.
     */
    public void printBaum(PrintStream o) {
        o.println(root.toString(0));
    }


    /*
     * encode(StringBuffer):
     * Komprimiert mit der vorhandenen Codetabelle
     * alle Zeichen kodieren.
     * Rckgabe: StringBuffer
     */
    public StringBuffer encode(StringBuffer data) {
	// speichert die aktuell kodierten Bits
        StringBuffer binary = new StringBuffer();
	// speichert die fertig komprimierten Daten
        StringBuffer encoded = new StringBuffer();

        // alle Zeichen einzeln durchgehen
        for(int i = 0; i < data.length(); i++) {
            int c = (int) data.charAt(i);
	    // den Binrcode aus der Codetabelle holen
            binary.append(codetable[c]);

	    // Wenn in binary mehr als acht Bits sind,
	    // kann ein Zeichen daraus gemacht werden.
            while (binary.length() >= 8) {
		// die ersten 8 Bits konvertieren
                int b = Integer.parseInt(binary.substring(0, 8), 2);
		// das resultierende Zeichen speichern
                encoded.append((char) b);
		// die ersten 8 Bits lschen
                binary.delete(0, 8);
            }
        }

        // Gibt es noch Bits zum Kodieren?
        if (binary.length() > 0) {

            // die Anzahl der Bits mssen ein Vielfaches von 8 sein
	    // deshalb noch abschlieende Einsen am Ende hinzufgen
            while(binary.length() % 8 != 0) binary.append('1');


            // jetzt noch die restlichen Zeichen umwandeln
	    while (binary.length() >= 8) {
                int b = Integer.parseInt(binary.substring(0, 8), 2);
                encoded.append((char) b);
                binary.delete(0,8);
	    }
        }

	// die komprimierten Daten zurckgeben
        return encoded;
    }


    /*
     * decode(StringBuffer, int):
     * Dekomprimiert mit dem vorhandenen Code-Baum.
     * Rckgabe: StringBuffer
     */
    public StringBuffer decode(StringBuffer data, int startindex) {
	// aktuelle Position im StringBuffer data.
	// Beginnt bei startindex (davor war die Codetabelle)
        int index = startindex;
	
	// speichert die Bits
        StringBuffer binary = new StringBuffer();
	// speichert die dekomprimierten Daten
        StringBuffer decoded = new StringBuffer();
	// zeigt immer auf den aktuellen Knoten
        Node n = root;
	// Position im StringBuffer binary
        int j = 0;

	// alle Zeichen durchgehen
        while(index < data.length()) {
            int c = (int) data.charAt(index++);
	    // in Binrcode konvertieren
            String code = Integer.toBinaryString(c);
            // fhrende Nullen wieder hinzufgen
            while(code.length() < 8) code = "0" + code;

	    // Binrcode bernehmen
            binary.append(code);

            // jetzt den Baum durchgehen
            while(j < binary.length()) {
                // entweder nach links oder nach rechts
                if (binary.charAt(j) == '0') {
                    n = n.getLeftChild();
                } else {
                    n = n.getRightChild();
                }

		// Position in binary um eins erhhen
                j++;

                // An einem Blatt angekommen?
                if (n.getLeftChild() == null && n.getRightChild() == null) {
                    // und sind noch zeichen zu dekodieren?
		    // oder sind die Daten schon komplett?
                    if (decoded.length() < characterscount) {
			// das dekodierte Zeichen speichern
                        decoded.append((char) n.getData());
                    }
                    // Zeiger wieder auf die Wurzel setzen
		    n = root;
		    // das Codewort lschen und Position wieder auf Anfang setzen
                    binary.delete(0, j);
                    j = 0;
                }
            }

        }

	// die dekomprimierten Daten zurckgeben
        return decoded;
    }
}

