/*
 * NetworkSpeed
 * (c) 2006 Andreas Dangel <adabolo@adabolo.de>
 * list.c   2006-10-18
 * 
 */

#include "list.h"
#include "main.h"
#include <stdio.h>
#include <stdlib.h>


// private function prototypes
static struct list * new_list_element();
static void copy_mac(u_char * dest, const u_char * src);
static int cmp_mac(const u_char * mac1, const u_char * mac2);
static void _free_list_recursive(struct list * p);

// global variable that stores the beginning of the linked list
static struct list * head = NULL;

/*
 * new_list_element():
 * 
 * Allocates memory for a new list entry and returns
 * the pointer to the new element.
 */
static struct list * new_list_element() {
    struct list * t = malloc(sizeof(struct list));
    
    if (t == NULL) fprintf(stderr, "new_list_element: malloc failed!\n");
    
    t->next = NULL;
    t->sent = 0;
    t->received = 0;
    return t;   
}

/*
 * copy_mac():
 * 
 * Copies ETHER_ADDR_LEN bytes from src to dest char array.
 * ETHER_ADDR_LEN is 6.
 */
static void copy_mac(u_char * dest, const u_char * src) {
    int i;
    
    for(i = 0; i < ETHER_ADDR_LEN; i++)
        dest[i] = src[i];
}

/*
 * cmp_mac():
 * 
 * Compares two mac addresses literally.
 * Returns 0, if the addresses are equal and -1 else.
 */
static int cmp_mac(const u_char * mac1, const u_char * mac2) {
    int i;
    
    for(i = 0; i < ETHER_ADDR_LEN; i++) {
        if (mac1[i] != mac2[i]) return -1;
    }
    
    return 0;
}

/*
 * init_list():
 * 
 * Creates the head element to create a new list.
 * Frees the existing list before that.
 */
void init_list() {
    if (head != NULL) {
        free_list(head);
    }
    
    head = new_list_element();
}

/*
 * packet_counter_sent():
 * 
 * Creates a new list element for the given mac address, if this element
 * doesn't exist. Adds the bytes to the sent variable in the element.
 */
void packet_counter_sent(const u_char ether_addr[], unsigned int bytes) {
    struct list * p;
            
    p = head;
    while(p->next != NULL) {
        p = p->next;
        if (p->ethernet_address != NULL && cmp_mac(p->ethernet_address, ether_addr) == 0) {
            p->sent += bytes;
            return;
        }
    }
    
    // the element doesn't exist, so we will create one
    p->next = new_list_element();
    copy_mac(p->next->ethernet_address, ether_addr);
    p->next->sent += bytes;
}

/*
 * packet_counter_received():
 * 
 * Creates a new list element for the given mac address, if this element
 * doesn't exist. Adds the bytes to the received variable in the element.
 */
void packet_counter_received(const u_char ether_addr[], unsigned int bytes) {
    struct list * p;
    
    p = head;
    while(p->next != NULL) {
        p = p->next;
        if (p->ethernet_address != NULL && cmp_mac(p->ethernet_address, ether_addr) == 0) {
            p->received += bytes;
            return;
        }
    }
    
    // the element doesn't exist, so we will create one
    p->next = new_list_element();
    copy_mac(p->next->ethernet_address, ether_addr);
    p->next->received += bytes;
}

/*
 * count():
 * 
 * Counts the number of list elements and returns it.
 */
unsigned int count() {
    struct list * p;
    unsigned int i = 0;
    
    p = head;
    while(p->next != NULL) {
        i++;
        p = p->next;
    }
    
    return i;
}

/*
 * get():
 * 
 * Returns a pointer to a specific list element. The first
 * element has index 0.
 */
struct list * get(int index) {
    struct list * p = head;
    int i;
    
    for(i = 0; i <= index; i++) p = p->next;
    
    return p;
}

/*
 * print_list():
 * 
 * Write the statistics to the given filename.
 * File format:
mac     sent        received
aa-aa-aa-aa-aa-aa   1.12    1.12
 *
 * The numbers are in KBit/s.
 */
void print_list(double seconds, const char * filename) {
    struct list * p;
    FILE * file;
    int i;
    
    file = fopen(filename, "w");
    if (file == NULL) {
        fprintf(stderr, "error writing file %s\n", filename);
        return;
    }
    
    fprintf(file, "mac\tsent\treceived\n");
    
    p = head;
    while(p->next != NULL) {
        p = p->next;
        for(i = 0; i < ETHER_ADDR_LEN-1; i++) fprintf(file, "%02X-", p->ethernet_address[i]);
        fprintf(file, "%02X\t%.2f\t%.2f\n", p->ethernet_address[ETHER_ADDR_LEN-1], p->sent*8/1000/seconds, p->received*8/1000/seconds);
    }
    
    fclose(file);
}

/*
 * clear_counters():
 * 
 * Sets the variables sent and received of each element to zero.
 */
void clear_counters() {
    struct list * p;
    
    p = head;
    
    while(p->next != NULL) {
        p = p->next;
        
        p->sent = 0;
        p->received = 0;
    }
}

/*
 * free_list():
 * 
 * Deletes every list element and frees the memory
 */
void free_list() {
    _free_list_recursive(head);
    head = NULL;
}


/*
 * _free_list_recursive():
 * 
 * A helper function that is called recursive to free all memory
 */
static void _free_list_recursive(struct list * p) {
    if (p->next != NULL) {
        _free_list_recursive(p->next);
    }
    
    free(p);
}




