/*
 * NetworkSpeed
 * (c) 2006 Andreas Dangel <adabolo@adabolo.de>
 * main.c   2006-10-18
 *
 * Das Programm soll die Pakete an einer Netzwerkkarte sniffen und
 * darstellen, welcher Rechner wie viel Bandbreite benutzt.
 * Es wird regelmig eine Statistik-Datei geschrieben, in der
 * alle MAC-Adressen aufgelistet sind mit der benutzen Bandbreite
 * in Sende- und Empfangsrichtung.
 * 
 * Am Anfang dieser Datei gibt es einige #define-Anweisungen,
 * ber die das Programm konfiguriert werden kann.
 * 
 *
 * siehe auch:
 * http://www.tcpdump.org/pcap.htm
 */

//##################################################
// Configuration...
#define VERSION_STRING ("v1.1")
#define COUNT_PACKETS 100
#define LOGFILE ("/var/log/networkspeed.log")
#define PIDFILE ("/var/run/networkspeed.pid")
#define STATFILE ("/tmp/networkspeed")
#define CAPTURE_FILTER ("")
//###################################################

#include <stdio.h>
#include <time.h>
#include <pcap.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdarg.h>
#include "main.h"
#include "list.h"


// prototypes
static void count_packet(u_char *args, const struct pcap_pkthdr *header, const u_char *packet);
static void print_log(const char * format, ...);
static void signal_handler(int signum);
static void install_signal_handlers();
static double diff_timeval(struct timeval t1, struct timeval t2);


// global variable for the capture handle
static pcap_t *handle = NULL;
// filehandle for the logfile
static FILE * logfile_handle = NULL;








/*
 * main():
 * 
 * The main function. Checks the arguments,
 * installs signal handler, forks into background,
 * opens the network device and starts capturing.
 */
int main(int argc, char* argv[]) {
    char *dev;                      // the network device
    char errbuf[PCAP_ERRBUF_SIZE];  // error string
    bpf_u_int32 mask;               // network mask
    bpf_u_int32 net;                // network address
    struct bpf_program fp;          // the capture filter
    struct in_addr addr;            // used to print netmask and netaddress
    struct timeval time1, time2;    // stores the time between the captured packets
    FILE * f_pid;                   // the pid file


    // syntax check: there must be exactly one argument: the network device
    if (argc != 2) {
        fprintf(stderr, "Usage: %s dev\n", argv[0]);
        return 1;
    }
    
    
    install_signal_handlers();
    
        
    printf("NetworkSpeed %s\n", VERSION_STRING);
    printf("*****************\n");
    printf("(c) 2006 Andreas Dangel\n");
    printf("\n");
    
    printf("Will update statistics after receiving %d packets\n", COUNT_PACKETS);
    printf("Will write statistics to file \"%s\"\n", STATFILE);
    printf("Logfile: %s\n", LOGFILE);
    printf("PID-File: %s\n", PIDFILE);
    printf("\n");
    
    
    dev = argv[1];        
    // find the properties for the device
    if (pcap_lookupnet(dev, &net, &mask, errbuf) == -1) {
        fprintf(stderr, "Couldn't get netmask for device %s: %s\n", dev, errbuf);
        return(2);
    }
    printf(" DEV: %s\n", dev);
    addr.s_addr = net;
    printf(" NET: %s\n", inet_ntoa(addr));
    addr.s_addr = mask;
    printf("MASK: %s\n", inet_ntoa(addr));



    // change working directory
    chdir("/");
        
    // open the logfile
    logfile_handle = fopen(LOGFILE, "a");
    print_log("NetworkSpeed %s started", VERSION_STRING);
    if (logfile_handle == NULL) {
        print_log("Warning! Couldn't open logfile %s", LOGFILE);
    }
    print_log("Capturing on device %s", dev);
    print_log("Using filter: %s", CAPTURE_FILTER);
    print_log("Child-PID=%d (%s)", getpid(), PIDFILE);
    print_log("Logfile: %s", LOGFILE);
    print_log("Statistics: %s", STATFILE);
    print_log("New statistics every %d packets", COUNT_PACKETS);

    // write the pid file
    f_pid = fopen(PIDFILE, "w");
    if (f_pid == NULL) {
        print_log("error while writing pid-file %s: %s", PIDFILE, strerror(errno));
    } else {
        fprintf(f_pid, "%d", getpid());
        fclose(f_pid);
    }
    
    // open the device in promiscous mode
    handle = pcap_open_live(dev, BUFSIZ, TRUE, 1000, errbuf);
    if (handle == NULL) {
        print_log("Couldn't open device %s: %s", dev, errbuf);
        return(2);
    }
    if (pcap_compile(handle, &fp, CAPTURE_FILTER, 1, net) == -1) {
        print_log("Couldn't parse filter %s: %s", CAPTURE_FILTER, pcap_geterr(handle));
        return(2);
    }
    if (pcap_setfilter(handle, &fp) == -1) {
        print_log("Couldn't install filter %s: %s", CAPTURE_FILTER, pcap_geterr(handle));
        return(2);
    }
    
    print_log("starting capture");

    // endless loop...
    while(TRUE) {
        // initialize the list, remove all existing entries
        init_list();
        // capture packets
        gettimeofday(&time1, NULL);
        pcap_loop(handle, COUNT_PACKETS, &count_packet, NULL);
        gettimeofday(&time2, NULL);
        // update statistics
        print_list(diff_timeval(time2, time1), STATFILE);
    }
    
    // we never reach this point, theoretically
    pcap_close(handle);
    return 0;
}









    
/*
 * count_packet():
 * 
 * Will be called for each captured packet.
 * Extends the list with the packet data
 * (mac address, packet length)
 */
static void count_packet(u_char *args, const struct pcap_pkthdr *header, const u_char *packet) {
    const struct sniff_ethernet *ethernet;
    
    ethernet = (struct sniff_ethernet *)(packet);
    
    packet_counter_sent(ethernet->ether_shost, header->len);
    packet_counter_received(ethernet->ether_dhost, header->len);
}

/*
 * print_log():
 * 
 * Write into the logfile or to stderr, if there is no logfile
 */
static void print_log(const char * format, ...) {
    va_list ap;
    time_t t;
    char * timestr;
    
    va_start(ap, format);
    t = time(NULL);
    timestr = ctime(&t);
    // remove the last char, the newline
    timestr[strlen(timestr)-1] = 0;
    
    if (logfile_handle == NULL) {
        fprintf(stderr, "%s ", timestr);
        vfprintf(stderr, format, ap);
        fprintf(stderr, "\n");
    } else {
        fprintf(logfile_handle, "%s ", timestr);
        vfprintf(logfile_handle, format, ap);
        fprintf(logfile_handle, "\n");
        fflush(logfile_handle);
    }
    
    va_end(ap);
}

/*
 * signal_handler():
 * 
 * This function will be called, when the program should be shutdown.
 * Close the capture handle and the log file handle.
 */
static void signal_handler(int signum) {
    char * signame;
    
    if (signum == SIGQUIT) signame = "SIGQUIT";
    else if (signum == SIGINT) signame = "SIGINT";
    else if (signum == SIGTERM) signame = "SIGTERM";
    else signame = "UNKNOWN SIGNAL";
    
    
    print_log("%s received!!! exiting...", signame);
 
    if (handle != NULL) {
        pcap_close(handle);
    }
    
    // pid-file lschen
    unlink(PIDFILE);
    
    fclose(logfile_handle);
    
    exit(0);
}

/*
 * install_signal_handler():
 * 
 * Installs the signal handler for
 * some signals.
 */
static void install_signal_handlers() {
    __sighandler_t ret;
    
    ret = signal(SIGQUIT, &signal_handler);    
    if (ret == SIG_ERR) {
        fprintf(stderr, "error while installing sighandler SIGQUIT!\n");
    }
    // STRG+C ---> SIGINT
    ret = signal(SIGINT, &signal_handler);    
    if (ret == SIG_ERR) {
        fprintf(stderr, "error while installing sighandler SIGINT!\n");
    }
    
    // SIGTERM --> default signal of kill(1)
    ret = signal(SIGTERM, &signal_handler);
    if (ret == SIG_ERR) {
        fprintf(stderr, "error while installing sighandler SIGTERM!\n");
    }
}

/*
 * diff_timeval():
 * 
 * Calculates the difference between to time points given as
 * struct timeval.
 * The function returns a double.
 */
static double diff_timeval(struct timeval t1, struct timeval t2) {
    double d1, d2;
    
    d1 = t1.tv_sec + t1.tv_usec/1000000.0;
    d2 = t2.tv_sec + t2.tv_usec/1000000.0;
    
    return (d1-d2);
}


