import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Vector;

public class Assembler {

	private static Parser parser;
	
	public static boolean assemble(StringBuffer in, short[] out, Vector labels)
	throws ParserException {
		Scanner s = new Scanner(in);
		parser = new Parser();
		Vector codetable = parser.parse(s);
		if (codetable != null) {
			for(int i = 0; i < codetable.size(); i++) {
				CodeElement ce = (CodeElement) codetable.get(i);
				short[] data = ce.getData();
				for(int j = ce.getIndex(), k = 0; k < data.length; k++, j++) {
					out[j] = data[k];
				}
			}
			
			if (labels != null) {
				labels.clear();
				labels.addAll(parser.getLabels());
			}
			
			return true;
		} else {
			return false;
		}
	}
	
	public static boolean assemble(File f, short[] out, Vector labels)
	throws ParserException, IOException {
		StringBuffer in = new StringBuffer();
		FileReader r = new FileReader(f);
		int c;
		while ((c = r.read()) != -1) {
			in.append((char) c);
		}
		r.close();
		
		return assemble(in, out, labels);
	}
	
	public static boolean assemble(File fin, File fout)
	throws ParserException, IOException {
		short[] mem = new short[4096];
		boolean b = assemble(fin, mem, null);
		if (b) {
			FileWriter out = new FileWriter(fout);
			for(int i = 0; i < mem.length; i++) {
				out.write((mem[i] & 0xFF00) >> 8);
				out.write(mem[i] & 0x00FF);
			}
			out.close();
			return true;
		} else {
			return false;
		}
	}
	
	
	public static void main(String args[])
	throws IOException {
		
		if (args.length != 2) {
			System.err.println("Syntax: Assembler in.file out.file");
			System.exit(1);
		}
		
		boolean b;
		
		try {
			System.out.println("Will now assemble " +args[0] + " into " + args[1]);
			b = assemble(new File(args[0]), new File(args[1]));
		} catch (ParserException e) {
			System.err.println(e.getMessage());
			b = false;
		}
		
		if (b) {
			System.out.println("Assembling successfully done.");
			System.exit(0);
		} else {
			System.exit(2);
		}		
	}
	
	
	
	public static interface Identifier {
		public static final int TYPE_LABEL = 1;
		public static final int TYPE_DEFINITION = 2;
		
		public short getIndex();
		public String getIdentifier();
		public int getType();
	}
	
	public static interface CodeElement {
		public short getSize() throws ParserException;
		public short[] getData() throws ParserException;
		public short getIndex();
	}
	
	public static class Statement implements CodeElement {
		private int token;
		private Argument argument;
		private short address;
		
		public Statement(int token, Argument argument, short a) {
			this.token = token;
			this.argument = argument;
			this.address = a;
		}
		
		public short getSize() {
			return 1;
		}
		
		public short getIndex() {
			return address;
		}
		
		public short[] getData()
		throws ParserException {
			short[] s = new short[1];
			switch (token) {
			case Scanner.T_KEYWORD_STO:
				s[0] = 0x0000;
				break;
			case Scanner.T_KEYWORD_LDA:
				s[0] = 0x1000;
				break;
			case Scanner.T_KEYWORD_BRZ:
				s[0] = 0x2000;
				break;
			case Scanner.T_KEYWORD_ADD:
				s[0] = 0x3000;
				break;
			case Scanner.T_KEYWORD_SUB:
				s[0] = 0x4000;
				break;
			case Scanner.T_KEYWORD_OR:
				s[0] = 0x5000;
				break;
			case Scanner.T_KEYWORD_AND:
				s[0] = 0x6000;
				break;
			case Scanner.T_KEYWORD_XOR:
				s[0] = 0x7000;
				break;
			case Scanner.T_KEYWORD_LDAR:
				s[0] = (short) 0x8000;
				break;
			case Scanner.T_KEYWORD_INCAR:
				s[0] = (short) 0x9000;
				break;
			case Scanner.T_KEYWORD_LDI:
				s[0] = (short) 0xA000;
				break;
			case Scanner.T_KEYWORD_STI:
				s[0] = (short) 0xB000;
				break;
			case Scanner.T_KEYWORD_CPI:
				s[0] = (short) 0xC000;
				break;
			case Scanner.T_KEYWORD_SHL:
				s[0] = (short) 0xD000;
				break;
			case Scanner.T_KEYWORD_SHR:
				s[0] = (short) 0xE000;
				break;
			case Scanner.T_KEYWORD_BRC:
				s[0] = (short) 0xF000;
				break;	
			}
			
			if (argument != null)
				s[0] += (short) (argument.getNumber() & 0x0FFF);
			
			return s;
		}
		
		public String toString() {
			if (argument != null) {
				try {
					return "Stmt: " + Scanner.getTokenDescription(token) + ", offset: " + address + ", arg: " + argument.getNumber();
				} catch (ParserException e) {
					return "Stmt: " + Scanner.getTokenDescription(token) + ", offest: " + address + ", arg: " + e.getMessage();
				}
			} else {
				return "Stmt: " + Scanner.getTokenDescription(token) + ", offset: " + address;
			}
		}
	}
	
	public static class Definition implements CodeElement, Identifier {
		private String identifier;
		private int keyword;
		private Vector arguments;
		private short address;
		
		public Definition(String id, int type, Vector args) {
			identifier = id;
			keyword = type;
			arguments = args;
		}
		
		public int getType() {
			return Identifier.TYPE_DEFINITION;
		}
		
		public short[] getData()
		throws ParserException {
			Vector data = new Vector();
			
			switch(keyword) {
			case Scanner.T_KEYWORD_DB:
				for(int i = 0; i < arguments.size(); i++) {
					Argument a = (Argument) arguments.get(i);
					if (a.getToken() == Scanner.T_STRING) {
						String s = a.getData().substring(1, a.getData().length()-1);
						if (s.length() % 2 != 0) s += "\0";
						for(int j = 0; j < s.length(); j++) {
							short b = 0;
							b += (short) ((s.charAt(j) & 0xFF) << 8);
							j++;
							b += (short) (s.charAt(j) & 0xFF);
							data.add(new Short(b));
						}						
					} else if (a.getToken() == Scanner.T_IDENTIFIER) {
						Definition d = parser.getDefinitionFromIdentifier(a.getData());
						if (d != null && d.getKeyword() == Scanner.T_KEYWORD_EQU) {
							short [] b = d.getData();
							for (int j = 0; j < b.length; j++) data.add(new Short(b[j]));
						} else {
							throw new ParserException("unknown identifier: " + a.getData());
						}						
					} else {
						short b = a.getNumber();
						if (b > 0xFF) {
							throw new ParserException("warning: value too big, identifier: " + identifier);
						}
						
						if (i % 2 == 0) {
							data.add(new Short((short) ((b & 0xFF) << 8)));
						} else {
							short c = ((Short) data.remove(data.size()-1)).shortValue();
							c += b & 0xFF;
							data.add(new Short(c));
						}
					}
				}
				break;
			case Scanner.T_KEYWORD_DC:
			case Scanner.T_KEYWORD_DW:
			case Scanner.T_KEYWORD_EQU:
				for(int i = 0; i < arguments.size(); i++) {
					Argument a = (Argument) arguments.get(i);
					if (a.getToken() == Scanner.T_STRING) {
						String s = a.getData().substring(1, a.getData().length()-1);
						if (s.length() % 2 != 0) s += "\0";
						for(int j = 0; j < s.length(); j++) {
							short b = 0;
							b += (short) ((s.charAt(j) & 0xFF) << 8);
							j++;
							b += (short) (s.charAt(j) & 0xFF);
							data.add(new Short(b));
						}
					} else if (a.getToken() == Scanner.T_IDENTIFIER) {
						Definition d = parser.getDefinitionFromIdentifier(a.getData());
						if (d != null && d.getKeyword() == Scanner.T_KEYWORD_EQU) {
							short [] b = d.getData();
							for (int j = 0; j < b.length; j++) data.add(new Short(b[j]));
						} else {
							throw new ParserException("unknown identifier: " + a.getData());
						}
					} else {
						data.add(new Short(a.getNumber()));
					}
				}
				break;
			}
			
			short[] s = new short[data.size()];
			for(int i = 0; i < s.length; i++) {
				s[i] = ((Short) data.get(i)).shortValue();
			}
			
			return s;
		}
		
		public void setAddress(short a) {
			address = a;
		}
		
		public short getSize() throws ParserException {
			return (short) getData().length;
		}
		
		public short getIndex() {
			return address;
		}
		
		public String getIdentifier() {
			return identifier;
		}
		
		public int getKeyword() {
			return keyword;
		}
		
		public String toString() {
			return "Def.: " + Scanner.getTokenDescription(keyword) + ", offset: " + address + ", id: " + identifier;
		}
	}
	
	public static class Label implements Identifier {
		private String identifier;
		private short address;
		public Label(String id, short a) {
			identifier = id;
			address = a;
		}
		
		public String toString() {
			return "LABEL: id=" + identifier + ", address=" + address;
		}
		
		public short getIndex() {
			return address;
		}
		
		public String getIdentifier() {
			return identifier;
		}
		
		public int getType() {
			return Identifier.TYPE_LABEL;
		}
	}
	
	public static class Argument {
		private String data;
		private int token;
		public Argument(String d, int type){
			data = d;
			token = type;
		}
		
		public String getData() { return data; }
		public int getToken() { return token; }
		
		public short getNumber() 
		throws ParserException {
			short s = -1;
			
			switch (token) {
			case Scanner.T_NUMBER_BINARY:
				s = (short) (Integer.parseInt(data.substring(0, data.length()-1), 2));
				break;
			case Scanner.T_NUMBER_DECIMAL:
				s = (short) (Integer.parseInt(data));
				break;
			case Scanner.T_NUMBER_HEX:
				s = (short) (Integer.parseInt(data.substring(1), 16));
				break;
			case Scanner.T_NUMBER_HEX_0X:
				s = (short) (Integer.parseInt(data.substring(2), 16));
				break;
			case Scanner.T_NUMBER_OCTAL:
				s = (short) (Integer.parseInt(data.substring(1), 8));
				break;
			case Scanner.T_IDENTIFIER:
				Definition d = parser.getDefinitionFromIdentifier(data);
				if (d != null) return d.getIndex();
				
				Label l = parser.getLabelFromIdentifier(data);
				if (l != null) return l.getIndex();
				
				throw new ParserException("Parser error: unknown identifier: " + data);
			}
			
			return s;
			
		}
		
		public String toString() {
			return "Argument: " + data + " (" + Scanner.getTokenDescription(token) + ")";
		}
	}
	
	public static class ParserException extends Exception {
		private static final long serialVersionUID = 1234567892L;
		
		public ParserException(String msg) {
			super(msg);
		}
	}
	
	public static class Parser {
		/*
		 * Parser:
		 * grammatic:
		 * 
		 * start    ::= whitespace_optional line start1
		 *   Firstmenge: WHITESPACE KEYWORD_INCLUDE, IDENTIFIER,
		 *     KEYWORD_STO, KEYWORD_LDA, KEYWORD_BRZ, KEYWORD_ADD, KEYWORD_SUB,
		 *     KEYWORD_OR, KEYWORD_AND, KEYWORD_XOR, KEWORD_LDAR, KEYWORD_INCAR,
		 *     KEYWORD_LDI, KEYWORD_STI, KEYWORD_CPI, KEYWORD_SHL, KEYWORD_SHR,
		 *     KEYWORD_BRC, KEYWORD_ORG, COMMENT, NEWLINE
		 * start1   ::= start
		 * start1   ::= epsilon
		 * line ::= include comment_optional NEWLINE
		 * 	 Firstmenge: KEYWORD_INCLUDE
		 * line ::= IDENTIFIER label_or_definition statement_optional whitespace_optional comment_optional NEWLINE
		 * line ::= statement comment_optional NEWLINE
		 *   Firstmenge:
		 *     KEYWORD_STO, KEYWORD_LDA, KEYWORD_BRZ, KEYWORD_ADD, KEYWORD_SUB,
		 *     KEYWORD_OR, KEYWORD_AND, KEYWORD_XOR, KEWORD_LDAR, KEYWORD_INCAR,
		 *     KEYWORD_LDI, KEYWORD_STI, KEYWORD_CPI, KEYWORD_SHL, KEYWORD_SHR,
		 *     KEYWORD_BRC,
		 *     KEYWORD_ORG
		 * line ::= COMMENT NEWLINE
		 * line ::= NEWLINE
		 * include ::= KEYWORD_INCLUDE WHITESPACE STRING whitespace_optional
		 * label_or_definition ::= COLON whitespace_optional
		 * label_or_definition ::= WHITESPACE definition2
		 * comment_optional ::= COMMENT
		 * comment_optional ::= epsilon
		 * statement_optional ::= statement
		 * statement_optional ::= epsilon
		 *   Firstmenge:
		 *     KEYWORD_STO, KEYWORD_LDA, KEYWORD_BRZ, KEYWORD_ADD, KEYWORD_SUB,
		 *     KEYWORD_OR, KEYWORD_AND, KEYWORD_XOR, KEWORD_LDAR, KEYWORD_INCAR,
		 *     KEYWORD_LDI, KEYWORD_STI, KEYWORD_CPI, KEYWORD_SHL, KEYWORD_SHR,
		 *     KEYWORD_BRC,
		 *     KEYWORD_ORG
		 * statement ::= KEYWORD_STO  WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_LDA  WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_BRZ  WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_ADD  WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_SUB  WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_OR   WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_AND  WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_XOR  WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_LDAR WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_INCAR whitespace_optional
		 * statement ::= KEYWORD_LDI whitespace_optional
		 * statement ::= KEYWORD_STI whitespace_optional
		 * statement ::= KEYWORD_CPI whitespace_optional
		 * statement ::= KEYWORD_SHL whitespace_optional
		 * statement ::= KEYWORD_SHR whitespace_optional
		 * statement ::= KEYWORD_BRC WHITESPACE argument whitespace_optional
		 * statement ::= KEYWORD_ORG WHITESPACE number whitespace_optional
		 * argument ::= number | IDENTIFIER
		 * definition2 ::= KEYWORD_DC WHITESPACE argument2
		 * definition2 ::= KEYWORD_DW WHITESPACE argument2
		 * definition2 ::= KEYWORD_DB WHITESPACE argument2
		 * definition2 ::= KEYWORD_EQU WHITESPACE argument2
		 * argument2 ::= number whitespace_optional argument3 whitespace_optional
		 * argument2 ::= STRING whitespace_optional argument3 whitespace_optional
		 * argument2 ::= IDENTIFIER whitespace_optional argument3 whitespace_optional
		 * argument3 ::= COMMA whitespace_optional argument2 whitespace_optional
		 * argument3 ::= epsilon
		 * number ::= NUMBER_HEX_0X | NUMBER_HEX | NUMBER_OCTAL | NUMBER_DECIMAL | NUMBER_BINARY
		 * whitespace_optional ::= WHITESPACE
		 * whitespace_optional ::= epsilon
		 */

		private int lookahead;
		private Scanner scanner;
		private Vector codetable;
		private boolean firstpass = true;
		private String last_identifier = null;
		private Vector arguments;
		private Vector definitions;
		private Vector labels;
		private short current_address = 0;
		
				
		public Definition getDefinitionFromIdentifier(String id) {
			for (int i = 0; i < definitions.size(); i++) {
				Definition d = (Definition) definitions.get(i);
				if (d.getIdentifier().equals(id)) return d;
			}
			
			return null;
		}
		
		public Label getLabelFromIdentifier(String id) {
			for (int i = 0; i < labels.size(); i++) {
				Label l = (Label) labels.get(i);
				if (l.getIdentifier().equals(id)) return l;
			}
			
			return null;
		}
		
		public Vector getLabels() {
			Vector v = new Vector();
			v.addAll(labels);
			v.addAll(definitions);
			return v;
		}
		
		public Vector parse(Scanner s)
		throws ParserException {		
			scanner = s;
			
			// First pass:
			// * collect equ statements
			// * collect all other definitions (identifiers)
			definitions = new Vector();
			firstpass = true;
			lookahead = scanner.getNextToken();
			if (start() == true) {
				// no errors
				// second pass:
				// * recognize org
				// * add statements and definitions (w/o equ) to code table
				// * collect the labels 
				firstpass = false;
				codetable = new Vector();
				labels = new Vector();
				current_address = 0;
				lookahead = scanner.getNextToken();
				if (start() == true) {
					// second pass also without error...
					return codetable;
				}
			}
			
			return null;
		}
		
		
		private boolean start()
		throws ParserException {
			return whitespace_optional() & line() & start1();
		}
		
		private boolean start1()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_WHITESPACE:
			case Scanner.T_KEYWORD_INCLUDE:
			case Scanner.T_IDENTIFIER:
			case Scanner.T_KEYWORD_STO:
			case Scanner.T_KEYWORD_LDA:
			case Scanner.T_KEYWORD_BRZ:
			case Scanner.T_KEYWORD_ADD:
			case Scanner.T_KEYWORD_SUB:
			case Scanner.T_KEYWORD_OR:
			case Scanner.T_KEYWORD_AND:
			case Scanner.T_KEYWORD_XOR:
			case Scanner.T_KEYWORD_LDAR:
			case Scanner.T_KEYWORD_INCAR:
			case Scanner.T_KEYWORD_LDI:
			case Scanner.T_KEYWORD_STI:
			case Scanner.T_KEYWORD_CPI:
			case Scanner.T_KEYWORD_SHL:
			case Scanner.T_KEYWORD_SHR:
			case Scanner.T_KEYWORD_BRC:
			case Scanner.T_KEYWORD_ORG:
			case Scanner.T_COMMENT:
			case Scanner.T_NEWLINE:
				return start();
			default:
				return true;
			}
		}

		private boolean line()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_KEYWORD_INCLUDE:
				return include() & whitespace_optional() & TERMINATOR(Scanner.T_NEWLINE);
			case Scanner.T_IDENTIFIER:
				return TERMINATOR(Scanner.T_IDENTIFIER) & label_or_definition()
				& statement_optional() & whitespace_optional()
				& comment_optional() & TERMINATOR(Scanner.T_NEWLINE);
			case Scanner.T_KEYWORD_STO:
			case Scanner.T_KEYWORD_LDA:
			case Scanner.T_KEYWORD_BRZ:
			case Scanner.T_KEYWORD_ADD:
			case Scanner.T_KEYWORD_SUB:
			case Scanner.T_KEYWORD_OR:
			case Scanner.T_KEYWORD_AND:
			case Scanner.T_KEYWORD_XOR:
			case Scanner.T_KEYWORD_LDAR:
			case Scanner.T_KEYWORD_INCAR:
			case Scanner.T_KEYWORD_LDI:
			case Scanner.T_KEYWORD_STI:
			case Scanner.T_KEYWORD_CPI:
			case Scanner.T_KEYWORD_SHL:
			case Scanner.T_KEYWORD_SHR:
			case Scanner.T_KEYWORD_BRC:
			case Scanner.T_KEYWORD_ORG:
				return statement() & comment_optional() & TERMINATOR(Scanner.T_NEWLINE);
			case Scanner.T_COMMENT:
				return TERMINATOR(Scanner.T_COMMENT) & TERMINATOR(Scanner.T_NEWLINE);
			case Scanner.T_NEWLINE:
				return TERMINATOR(Scanner.T_NEWLINE);
			default:
				showError("unexpected token (line()): " + Scanner.getTokenDescription(lookahead));
				return false;
			}
		}
	
		private boolean include()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_KEYWORD_INCLUDE:
				return TERMINATOR(Scanner.T_KEYWORD_INCLUDE)
					& TERMINATOR(Scanner.T_WHITESPACE)
					& TERMINATOR(Scanner.T_STRING)
					& whitespace_optional();
			default:
				showError("unexpected token (include()): " + Scanner.getTokenDescription(lookahead)
						+ "; expected " + Scanner.getTokenDescription(Scanner.T_KEYWORD_INCLUDE));
				return false;
			}
		}
		
		private boolean label_or_definition()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_COLON:
				// label
				if (!firstpass) {
					labels.add(new Label(last_identifier, current_address));
				}
				return TERMINATOR(Scanner.T_COLON) & whitespace_optional();
			case Scanner.T_WHITESPACE:
				// definition
				return TERMINATOR(Scanner.T_WHITESPACE) & definition2();
			default:
				showError("unexpected token (label_or_definition()): " + Scanner.getTokenDescription(lookahead)
						+ "; expected " + Scanner.getTokenDescription(Scanner.T_COLON) + " or "
						+ Scanner.getTokenDescription(Scanner.T_WHITESPACE));
				return false;
			}
		}
		
		private boolean comment_optional()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_COMMENT:
				return TERMINATOR(Scanner.T_COMMENT);
			default:
				return true;
			}
		}
		
		private boolean statement_optional()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_KEYWORD_STO:
			case Scanner.T_KEYWORD_LDA:
			case Scanner.T_KEYWORD_BRZ:
			case Scanner.T_KEYWORD_ADD:
			case Scanner.T_KEYWORD_SUB:
			case Scanner.T_KEYWORD_OR:
			case Scanner.T_KEYWORD_AND:
			case Scanner.T_KEYWORD_XOR:
			case Scanner.T_KEYWORD_LDAR:
			case Scanner.T_KEYWORD_INCAR:
			case Scanner.T_KEYWORD_LDI:
			case Scanner.T_KEYWORD_STI:
			case Scanner.T_KEYWORD_CPI:
			case Scanner.T_KEYWORD_SHL:
			case Scanner.T_KEYWORD_SHR:
			case Scanner.T_KEYWORD_BRC:
			case Scanner.T_KEYWORD_ORG:
				return statement();
			default:
				return true;
			}
		}

		private boolean statement()
		throws ParserException {
			boolean b;
			arguments = new Vector();
			switch (lookahead) {
			case Scanner.T_KEYWORD_STO:
				b = TERMINATOR(Scanner.T_KEYWORD_STO)
					& TERMINATOR(Scanner.T_WHITESPACE)
					& argument()
					& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_STO, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_LDA:
				b = TERMINATOR(Scanner.T_KEYWORD_LDA)
					& TERMINATOR(Scanner.T_WHITESPACE)
					& argument()
					& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_LDA, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_BRZ:
				b = TERMINATOR(Scanner.T_KEYWORD_BRZ)
				& TERMINATOR(Scanner.T_WHITESPACE)
				& argument()
				& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_BRZ, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_ADD:
				b = TERMINATOR(Scanner.T_KEYWORD_ADD)
				& TERMINATOR(Scanner.T_WHITESPACE)
				& argument()
				& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_ADD, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_SUB:
				b = TERMINATOR(Scanner.T_KEYWORD_SUB)
				& TERMINATOR(Scanner.T_WHITESPACE)
				& argument()
				& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_SUB, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_OR:
				b = TERMINATOR(Scanner.T_KEYWORD_OR)
				& TERMINATOR(Scanner.T_WHITESPACE)
				& argument()
				& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_OR, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_AND:
				b = TERMINATOR(Scanner.T_KEYWORD_AND)
				& TERMINATOR(Scanner.T_WHITESPACE)
				& argument()
				& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_AND, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_XOR:
				b = TERMINATOR(Scanner.T_KEYWORD_XOR)
				& TERMINATOR(Scanner.T_WHITESPACE)
				& argument()
				& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_XOR, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_LDAR:
				b = TERMINATOR(Scanner.T_KEYWORD_LDAR)
				& TERMINATOR(Scanner.T_WHITESPACE)
				& argument()
				& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_LDAR, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_INCAR:
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_INCAR, null, current_address++));
				return TERMINATOR(Scanner.T_KEYWORD_INCAR)
				& whitespace_optional();
			case Scanner.T_KEYWORD_LDI:
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_LDI, null, current_address++));
				return TERMINATOR(Scanner.T_KEYWORD_LDI)
				& whitespace_optional();
			case Scanner.T_KEYWORD_STI:
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_STI, null, current_address++));
				return TERMINATOR(Scanner.T_KEYWORD_STI)
				& whitespace_optional();
			case Scanner.T_KEYWORD_CPI:
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_CPI, null, current_address++));
				return TERMINATOR(Scanner.T_KEYWORD_CPI)
				& whitespace_optional();
			case Scanner.T_KEYWORD_SHL:
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_SHL, null, current_address++));
				return TERMINATOR(Scanner.T_KEYWORD_SHL)
				& whitespace_optional();
			case Scanner.T_KEYWORD_SHR:
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_SHR, null, current_address++));
				return TERMINATOR(Scanner.T_KEYWORD_SHR)
				& whitespace_optional();
			case Scanner.T_KEYWORD_BRC:
				b = TERMINATOR(Scanner.T_KEYWORD_BRC)
				& TERMINATOR(Scanner.T_WHITESPACE)
				& argument()
				& whitespace_optional();
				if (!firstpass)	codetable.add(new Statement(Scanner.T_KEYWORD_BRC, (Argument) arguments.get(0), current_address++));
				return b;
			case Scanner.T_KEYWORD_ORG:
				b = TERMINATOR(Scanner.T_KEYWORD_ORG)
				& TERMINATOR(Scanner.T_WHITESPACE)
				& argument()
				& whitespace_optional();
				if (!firstpass)	current_address = ((Argument) arguments.get(0)).getNumber();
				return b;
			default:
				showError("unexpected token (statement()): " + Scanner.getTokenDescription(lookahead) + "; statement expected");
				return false;
			}
		}

		private boolean argument()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_NUMBER_BINARY:
			case Scanner.T_NUMBER_DECIMAL:
			case Scanner.T_NUMBER_HEX:
			case Scanner.T_NUMBER_HEX_0X:
			case Scanner.T_NUMBER_OCTAL:
				arguments.add(new Argument(scanner.getyytext(), lookahead));
				return number();
			case Scanner.T_IDENTIFIER:
				arguments.add(new Argument(scanner.getyytext(), lookahead));
				return TERMINATOR(Scanner.T_IDENTIFIER);
			default:
				showError("unexpected token (argument()): " + Scanner.getTokenDescription(lookahead) + "; number, string or identifier expected");
				return false;
			}
		}

		private boolean definition2()
		throws ParserException {
			boolean b;
			String s = null;
			switch (lookahead) {
			case Scanner.T_KEYWORD_DC:
				arguments = new Vector();
				s = last_identifier;
				b = TERMINATOR(Scanner.T_KEYWORD_DC)
					& TERMINATOR(Scanner.T_WHITESPACE)
					& argument2();
				if (firstpass) definitions.add(new Definition(s, Scanner.T_KEYWORD_DC, arguments));
				if (!firstpass) {
					codetable.add(getDefinitionFromIdentifier(s));
					getDefinitionFromIdentifier(s).setAddress(current_address);
					current_address += getDefinitionFromIdentifier(s).getSize();
				}
				return b;
			case Scanner.T_KEYWORD_DW:
				arguments = new Vector();
				s = last_identifier;
				b = TERMINATOR(Scanner.T_KEYWORD_DW)
					& TERMINATOR(Scanner.T_WHITESPACE)
					& argument2();
				if (firstpass) definitions.add(new Definition(s, Scanner.T_KEYWORD_DW, arguments));
				if (!firstpass) {
					codetable.add(getDefinitionFromIdentifier(s));
					getDefinitionFromIdentifier(s).setAddress(current_address);
					current_address += getDefinitionFromIdentifier(s).getSize();
				}
				return b;
			case Scanner.T_KEYWORD_DB:
				arguments = new Vector();
				s = last_identifier;
				b = TERMINATOR(Scanner.T_KEYWORD_DB)
					& TERMINATOR(Scanner.T_WHITESPACE)
					& argument2();
				if (firstpass) definitions.add(new Definition(s, Scanner.T_KEYWORD_DB, arguments));
				if (!firstpass) {
					codetable.add(getDefinitionFromIdentifier(s));
					getDefinitionFromIdentifier(s).setAddress(current_address);
					current_address += getDefinitionFromIdentifier(s).getSize();
				}
				return b;
			case Scanner.T_KEYWORD_EQU:
				arguments = new Vector();
				s = last_identifier;
				b = TERMINATOR(Scanner.T_KEYWORD_EQU)
					& TERMINATOR(Scanner.T_WHITESPACE)
					& argument2();
				if (firstpass) definitions.add(new Definition(s, Scanner.T_KEYWORD_EQU, arguments));
				return b;
			default:
				showError("unexpected token (definition2()): " + Scanner.getTokenDescription(lookahead) + "; expected dc, dw, db, or equ");
			    return false;
			}
		}

		private boolean argument2()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_NUMBER_BINARY:
			case Scanner.T_NUMBER_DECIMAL:
			case Scanner.T_NUMBER_HEX:
			case Scanner.T_NUMBER_HEX_0X:
			case Scanner.T_NUMBER_OCTAL:
				arguments.add(new Argument(scanner.getyytext(), lookahead));
				return number()
					& whitespace_optional() & argument3() & whitespace_optional();
			case Scanner.T_STRING:
				arguments.add(new Argument(scanner.getyytext(), lookahead));
				return TERMINATOR(Scanner.T_STRING)
					& whitespace_optional() & argument3() & whitespace_optional();
			case Scanner.T_IDENTIFIER:
				arguments.add(new Argument(scanner.getyytext(), lookahead));
				return TERMINATOR(Scanner.T_IDENTIFIER)
					& whitespace_optional() & argument3() & whitespace_optional();
			default:
				showError("unexpected token (argument2()): " + Scanner.getTokenDescription(lookahead) + "; expected number, string or identifier");
				return false;
			}
		}

		private boolean argument3()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_COMMA:
				return TERMINATOR(Scanner.T_COMMA)
					& whitespace_optional() & argument2() & whitespace_optional();
			default:
				return true;
			}
		}
		
		private boolean number()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_NUMBER_HEX_0X:
				return TERMINATOR(Scanner.T_NUMBER_HEX_0X);
			case Scanner.T_NUMBER_HEX:
				return TERMINATOR(Scanner.T_NUMBER_HEX);
			case Scanner.T_NUMBER_OCTAL:
				return TERMINATOR(Scanner.T_NUMBER_OCTAL);
			case Scanner.T_NUMBER_DECIMAL:
				return TERMINATOR(Scanner.T_NUMBER_DECIMAL);
			case Scanner.T_NUMBER_BINARY:
				return TERMINATOR(Scanner.T_NUMBER_BINARY);
			default:
				showError("unexpected token (number()): " + Scanner.getTokenDescription(lookahead)
						+ "; expected a number");
				return false;
			}
		}
		
		private boolean whitespace_optional()
		throws ParserException {
			switch (lookahead) {
			case Scanner.T_WHITESPACE:
				return TERMINATOR(Scanner.T_WHITESPACE);
			default:
				return true;
			}
		}
		
		private boolean TERMINATOR(int t)
		throws ParserException {
			if (lookahead == t) {
				if (lookahead == Scanner.T_IDENTIFIER) {
					last_identifier = scanner.getyytext();
				}
				lookahead = scanner.getNextToken();
				return true;
			} else {
				showError(Scanner.getTokenDescription(t) + " expected");
				return false;
			}
		}
		
		private void showError(String msg)
		throws ParserException {
			String s = "Parser error: line " + scanner.getLine() + ", column "
			+ scanner.getColumn()+ ": " + msg;
			
			throw new ParserException(s);
		}
	}
	
	public static class Scanner {
		public static final int T_EOF = -2;
		public static final int T_SCANNER_ERROR = -1;
		public static final int T_UNKNOWN = 0;
		public static final int T_COMMENT = 1;
		public static final int T_NEWLINE = 2;
		public static final int T_WHITESPACE = 3;
		public static final int T_COLON = 4;
		public static final int T_COMMA = 5;
		public static final int T_KEYWORD_ORG = 6;
		public static final int T_KEYWORD_DC = 7;
		public static final int T_KEYWORD_DW = 8;
		public static final int T_KEYWORD_DB = 9;
		public static final int T_KEYWORD_EQU = 10;
		public static final int T_NUMBER_HEX_0X = 11;
		public static final int T_NUMBER_HEX = 12;
		public static final int T_NUMBER_OCTAL = 13;
		public static final int T_NUMBER_DECIMAL = 14;
		public static final int T_NUMBER_BINARY = 15;
		public static final int T_STRING = 16;
		public static final int T_KEYWORD_STO = 17;
		public static final int T_KEYWORD_LDA = 18;
		public static final int T_KEYWORD_BRZ = 19;
		public static final int T_KEYWORD_ADD = 20;
		public static final int T_KEYWORD_SUB = 21;
		public static final int T_KEYWORD_OR = 22;
		public static final int T_KEYWORD_AND = 23;
		public static final int T_KEYWORD_XOR = 24;
		public static final int T_KEYWORD_LDAR = 25;
		public static final int T_KEYWORD_INCAR = 26;
		public static final int T_KEYWORD_LDI = 27;
		public static final int T_KEYWORD_STI = 28;
		public static final int T_KEYWORD_CPI = 29;
		public static final int T_KEYWORD_SHL = 30;
		public static final int T_KEYWORD_SHR = 31;
		public static final int T_KEYWORD_BRC = 32;
		public static final int T_KEYWORD_INCLUDE = 33;
		public static final int T_IDENTIFIER = 34;
		
		
		private int line = 1;
		private int column = 0;
		private int index = 0;
		private int state = 0;
		private StringBuffer buffer = null;
		private String yytext = null;
		
		public Scanner(StringBuffer sb) {
			buffer = sb;
			buffer.append('\n');
		}
		
		public static String getTokenDescription(int t) {
			switch (t) {
			case T_EOF: return "T_EOF";
			case T_SCANNER_ERROR: return "T_SCANNER_ERROR";
			case T_UNKNOWN: return "T_UNKNOWN";
			case T_COMMENT: return "T_COMMENT";
			case T_NEWLINE: return "T_NEWLINE";
			case T_WHITESPACE: return "T_WHITESPACE";
			case T_COLON: return "T_COLON";
			case T_COMMA: return "T_COMMA";
			case T_KEYWORD_ORG: return "T_KEYWORD_ORG";
			case T_KEYWORD_DC: return "T_KEYWORD_DC";
			case T_KEYWORD_DW: return "T_KEYWORD_DW";
			case T_KEYWORD_DB: return "T_KEYWORD_DB";
			case T_KEYWORD_EQU: return "T_KEYWORD_EQU";
			case T_NUMBER_HEX_0X: return "T_NUMBER_HEX_0X";
			case T_NUMBER_HEX: return "T_NUMBER_HEX";
			case T_NUMBER_OCTAL: return "T_NUMBER_OCTAL";
			case T_NUMBER_DECIMAL: return "T_NUMBER_DECIMAL";
			case T_NUMBER_BINARY: return "T_NUMBER_BINARY";
			case T_STRING: return "T_STRING";
			case T_KEYWORD_STO: return "T_KEYWORD_STO";
			case T_KEYWORD_LDA: return "T_KEYWORD_LDA";
			case T_KEYWORD_BRZ: return "T_KEYWORD_BRZ";
			case T_KEYWORD_ADD: return "T_KEYWORD_ADD";
			case T_KEYWORD_SUB: return "T_KEYWORD_SUB";
			case T_KEYWORD_OR: return "T_KEYWORD_OR";
			case T_KEYWORD_AND: return "T_KEYWORD_AND";
			case T_KEYWORD_XOR: return "T_KEYWORD_XOR";
			case T_KEYWORD_LDAR: return "T_KEYWORD_LDAR";
			case T_KEYWORD_INCAR: return "T_KEYWORD_INCAR";
			case T_KEYWORD_LDI: return "T_KEYWORKD_LDI";
			case T_KEYWORD_STI: return "T_KEYWORD_STI";
			case T_KEYWORD_CPI: return "T_KEYWORD_CPI";
			case T_KEYWORD_SHL: return "T_KEYWORD_SHL";
			case T_KEYWORD_SHR: return "T_KEYWORD_SHR";
			case T_KEYWORD_BRC: return "T_KEYWORD_BRC";
			case T_KEYWORD_INCLUDE: return "T_KEYWORD_INCLUDE";
			case T_IDENTIFIER: return "T_IDENTIFIER";			
			}
			return "unknown token";
		}
		
		public int getLine() { return line; }
		public int getColumn() { return column; }
		public String getyytext() { return yytext; }
		
		public int getNextToken() {
			if (index == buffer.length()) {
				index = 0;
				line = 1;
				column = 0;
				state = 0;
				return T_EOF;
			}

			char c = buffer.charAt(index++);
			column++;
			
			switch (state) {
			case 0:
				yytext = new String();
				switch (c) {
				case ';':
					yytext += c;
					state = 1;
					return getNextToken();
				case '/':
					yytext += c;
					state = 2;
					return getNextToken();
				case '\n':
					yytext += c;
					state = 0;
					line++;
					column = 0;
					return T_NEWLINE;
				case '\t':
				case ' ':
					yytext += c;
					state = 3;
					return getNextToken();
				case ':':
					yytext += c;
					state = 0;
					return T_COLON;
				case ',':
					yytext += c;
					state = 0;
					return T_COMMA;
				case 'o':
				case 'O':
					yytext += c;
					state = 4;
					return getNextToken();
				case 'd':
				case 'D':
					yytext += c;
					state = 5;
					return getNextToken();
				case 'e':
				case 'E':
					yytext += c;
					state = 6;
					return getNextToken();
				case '0':
					yytext += c;
					state = 7;
					return getNextToken();
				case '$':
					yytext += c;
					state = 8;
					return getNextToken();
				case '1':
					yytext += c;
					state = 9;
					return getNextToken();
				case '\'':
					yytext += c;
					state = 10;
					return getNextToken();
				case '"':
					yytext += c;
					state = 11;
					return getNextToken();
				case 's':
				case 'S':
					yytext += c;
					state = 12;
					return getNextToken();
				case 'l':
				case 'L':
					yytext += c;
					state = 13;
					return getNextToken();
				case 'b':
				case 'B':
					yytext += c;
					state = 14;
					return getNextToken();
				case 'a':
				case 'A':
					yytext += c;
					state = 15;
					return getNextToken();
				case 'x':
				case 'X':
					yytext += c;
					state = 16;
					return getNextToken();
				case 'i':
				case 'I':
					yytext += c;
					state = 17;
					return getNextToken();
				case 'c':
				case 'C':
					yytext += c;
					state = 18;
					return getNextToken();
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					yytext += c;
					state = 21;
					return getNextToken();
				default:
					yytext += c;
					state = 0;
					return T_UNKNOWN;
				}
				
			case 1:
				switch (c) {
				case '\n':
					index--;
					state = 0;
					return T_COMMENT;
				default:
					yytext += c;
					state = 1;
					return getNextToken();
				}
				
			case 2:
				switch (c) {
				case '/':
					yytext += c;
					state = 1;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_UNKNOWN;
				}
				
			case 3:
				switch (c) {
				case ' ':
				case '\t':
					yytext += c;
					state = 3;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_WHITESPACE;
				}
				
			case 4:
				switch (c) {
				case 'r':
				case 'R':
					yytext += c;
					state = 22;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				/*
				case ';':
				case '/':
				case '\t':
				case ' ':
				case '\n':
					index--;
					state = 0;
					return T_IDENTIFIER;
				*/	
				}
				
			case 5:
				switch (c) {
				case 'c':
				case 'C':
					yytext += c;
					state = 0;
					return T_KEYWORD_DC;
				case 'w':
				case 'W':
					yytext += c;
					state = 0;
					return T_KEYWORD_DW;
				case 'b':
				case 'B':
					yytext += c;
					state = 0;
					return T_KEYWORD_DB;
				case 'a':
				case 'A':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;

				}
				
			case 6:
				switch (c) {
				case 'q':
				case 'Q':
					yytext += c;
					state = 23;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 7:
				switch (c) {
				case '0':
				case '1':
					yytext += c;
					state = 24;
					return getNextToken();
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
					yytext += c;
					state = 20;
					return getNextToken();
				case '7':
				case '8':
				case '9':
					yytext += c;
					state = 21;
					return getNextToken();
				case 'b':
				case 'B':
					yytext += c;
					state = 0;
					return T_NUMBER_BINARY;
				case 'x':
				case 'X':
					yytext += c;
					state = 25;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_NUMBER_DECIMAL;
				}
				
			case 8:
				switch (c) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
					yytext += c;
					state = 39;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_UNKNOWN;
				}
				
			case 9:
				switch (c) {
				case '0':
				case '1':
					yytext += c;
					state = 37;
					return getNextToken();
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					yytext += c;
					state = 21;
					return getNextToken();
				case 'b':
				case 'B':
					yytext += c;
					state = 0;
					return T_NUMBER_BINARY;
				default:
					index--;
					state = 0;
					return T_NUMBER_DECIMAL;
				}
				
			case 10:
				switch (c) {
				case '\'':
					yytext += c;
					state = 0;
					return T_STRING;
				case '\n':
					index -= yytext.length();
					yytext = yytext.substring(0, 1);
					state = 0;
					return T_UNKNOWN;
				default:
					yytext += c;
					state = 10;
					return getNextToken();
				}
				
			case 11:
				switch (c) {
				case '"':
					yytext += c;
					state = 0;
					return T_STRING;
				case '\n':
					index -= yytext.length();
					yytext = yytext.substring(0, 1);
					state = 0;
					return T_UNKNOWN;
				default:
					yytext += c;
					state = 11;
					return getNextToken();
				}
				
			case 12:
				switch (c) {
				case 't':
				case 'T':
					yytext += c;
					state = 26;
					return getNextToken();
				case 'u':
				case 'U':
					yytext += c;
					state = 27;
					return getNextToken();
				case 'h':
				case 'H':
					yytext += c;
					state = 28;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;				
				}
				
			case 13:
				switch (c) {
				case 'd':
				case 'D':
					yytext += c;
					state = 29;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 14:
				switch (c) {
				case 'r':
				case 'R':
					yytext += c;
					state = 30;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 15:
				switch (c) {
				case 'd':
				case 'D':
					yytext += c;
					state = 31;
					return getNextToken();
				case 'n':
				case 'N':
					yytext += c;
					state = 32;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 16:
				switch (c) {
				case 'o':
				case 'O':
					yytext += c;
					state = 33;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 17:
				switch (c) {
				case 'n':
				case 'N':
					yytext += c;
					state = 34;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 18:
				switch (c) {
				case 'p':
				case 'P':
					yytext += c;
					state = 35;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 19:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 20:
				switch (c) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
					yytext += c;
					state = 20;
					return getNextToken();
				case '7':
				case '8':
				case '9':
					yytext += c;
					state = 21;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_NUMBER_OCTAL;
				}
				
			case 21:
				switch (c) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					yytext += c;
					state = 21;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_NUMBER_DECIMAL;
				}
				
			case 22:
				switch (c) {
				case 'g':
				case 'G':
					yytext += c;
					state = 36;
					return getNextToken();				
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_OR;
				}
				
			case 23:
				switch (c) {
				case 'u':
				case 'U':
					yytext += c;
					state = 41;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;	
				}
				
			case 24:
				switch (c) {
				case '0':
				case '1':
					yytext += c;
					state = 24;
					return getNextToken();
				case 'b':
				case 'B':
					yytext += c;
					state = 0;
					return T_NUMBER_BINARY;
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
					yytext += c;
					state = 20;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_NUMBER_OCTAL;
				}
				
			case 25:
				switch (c) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
					yytext += c;
					state = 38;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_UNKNOWN;
				}
				
			case 26:
				switch (c) {
				case 'o':
				case 'O':
					yytext += c;
					state = 42;
					return getNextToken();
				case 'i':
				case 'I':
					yytext += c;
					state = 43;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 27:
				switch (c) {
				case 'b':
				case 'B':
					yytext += c;
					state = 44;
					return getNextToken();
				case 'a':
				case 'A':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 28:
				switch (c) {
				case 'r':
				case 'R':
					yytext += c;
					state = 45;
					return getNextToken();
				case 'l':
				case 'L':
					yytext += c;
					state = 46;
					return getNextToken();					
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 29:
				switch (c) {
				case 'a':
				case 'A':
					yytext += c;
					state = 40;
					return getNextToken();
				case 'i':
				case 'I':
					yytext += c;
					state = 47;
					return getNextToken();
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;					
				}
				
			case 30:
				switch (c) {
				case 'z':
				case 'Z':
					yytext += c;
					state = 48;
					return getNextToken();
				case 'c':
				case 'C':
					yytext += c;
					state = 49;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 31:
				switch (c) {
				case 'd':
				case 'D':
					yytext += c;
					state = 50;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 32:
				switch (c) {
				case 'd':
				case 'D':
					yytext += c;
					state = 51;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 33:
				switch (c) {
				case 'r':
				case 'R':
					yytext += c;
					state = 52;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 34:
				switch (c) {
				case 'c':
				case 'C':
					yytext += c;
					state = 53;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 35:
				switch (c) {
				case 'i':
				case 'I':
					yytext += c;
					state = 54;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 36:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_ORG;
				}
				
			case 37:
				switch (c) {
				case '0':
				case '1':
					yytext += c;
					state = 37;
					return getNextToken();
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					yytext += c;
					state = 21;
					return getNextToken();
				case 'b':
				case 'B':
					yytext += c;
					state = 0;
					return T_NUMBER_BINARY;
				default:
					index--;
					state = 0;
					return T_NUMBER_DECIMAL;
				}
				
			case 38:
				switch (c) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
					yytext += c;
					state = 38;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_NUMBER_HEX_0X;
				}
				
			case 39:
				switch (c) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
					yytext += c;
					state = 39;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_NUMBER_HEX;
				}
				
			case 40:
				switch (c) {
				case 'r':
				case 'R':
					yytext += c;
					state = 55;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_LDA;
				}
				
			case 41:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_EQU;
				}
				
			case 42:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_STO;
				}
				
			case 43:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_STI;
				}
				
			case 44:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_SUB;
				}
				
			case 45:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_SHR;
				}
				
			case 46:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_SHL;
				}
				
			case 47:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_LDI;
				}
				
			case 48:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_BRZ;
				}
				
			case 49:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_BRC;
				}
				
			case 50:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_ADD;
				}
				
			case 51:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_AND;
				}
				
			case 52:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_XOR;
				}
				
			case 53:
				switch (c) {
				case 'a':
				case 'A':
					yytext += c;
					state = 56;
					return getNextToken();
				case 'l':
				case 'L':
					yytext += c;
					state = 58;
					return getNextToken();
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 54:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_CPI;
				}
				
			case 55:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_LDAR;
				}
				
			case 56:
				switch (c) {
				case 'r':
				case 'R':
					yytext += c;
					state = 57;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;					
				}
				
			case 57:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_INCAR;
				}

			case 58:
				switch (c) {
				case 'u':
				case 'U':
					yytext += c;
					state = 59;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 59:
				switch (c) {
				case 'd':
				case 'D':
					yytext += c;
					state = 60;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 60:
				switch (c) {
				case 'e':
				case 'E':
					yytext += c;
					state = 61;
					return getNextToken();
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_IDENTIFIER;
				}
				
			case 61:
				switch (c) {
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
				case 'e':
				case 'E':
				case 'f':
				case 'F':
				case 'g':
				case 'G':
				case 'h':
				case 'H':
				case 'i':
				case 'I':
				case 'j':
				case 'J':
				case 'k':
				case 'K':
				case 'l':
				case 'L':
				case 'm':
				case 'M':
				case 'n':
				case 'N':
				case 'o':
				case 'O':
				case 'p':
				case 'P':
				case 'q':
				case 'Q':
				case 'r':
				case 'R':
				case 's':
				case 'S':
				case 't':
				case 'T':
				case 'u':
				case 'U':
				case 'v':
				case 'V':
				case 'w':
				case 'W':
				case 'x':
				case 'X':
				case 'y':
				case 'Y':
				case 'z':
				case 'Z':
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '_':
					yytext += c;
					state = 19;
					return getNextToken();
				default:
					index--;
					state = 0;
					return T_KEYWORD_INCLUDE;				
				}
				
			default:
				return T_SCANNER_ERROR;
			}
		}
		
	}
}
