import java.awt.BorderLayout;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.filechooser.FileFilter;

public class Editor extends JFrame implements ActionListener, KeyListener, WindowListener, CaretListener {
	private static final long serialVersionUID = 1234567894L;

	private static final String title = "ToySimulator - Editor";
	private MainFrame mf = null;
	private JTextArea text = new JTextArea(25, 80);
	private boolean saved = true;
	private String filename = null;
	private JLabel statusbar = new JLabel(" ");
	
	public Editor(MainFrame m) {
		super(title);
		mf = m;
		
		JPanel contentpane = new JPanel();
		contentpane.setLayout(new BorderLayout());
		
		contentpane.add(new JScrollPane(text), BorderLayout.CENTER);
		contentpane.add(statusbar, BorderLayout.SOUTH);
		
		text.addKeyListener(this);
		text.addCaretListener(this);
		text.setFont(new Font("monospaced", 0, 12));
		updateStatusbar();
		
		JMenuBar menubar = new JMenuBar();
		JMenu file_menu = new JMenu("File");
		file_menu.setMnemonic(KeyEvent.VK_F);
		JMenuItem file_new = new JMenuItem("New");
		file_new.setMnemonic(KeyEvent.VK_N);
		file_new.addActionListener(this);
		
		JMenuItem file_open = new JMenuItem("Open");
		file_open.addActionListener(this);
		file_open.setMnemonic(KeyEvent.VK_O);
		JMenuItem file_save = new JMenuItem("Save");
		file_save.addActionListener(this);
		file_save.setMnemonic(KeyEvent.VK_S);
		JMenuItem file_save_as = new JMenuItem("Save As");
		file_save_as.addActionListener(this);
		file_save_as.setMnemonic(KeyEvent.VK_A);
		JMenuItem file_assemble = new JMenuItem("Assemble and load into simulator");
		file_assemble.addActionListener(this);
		file_assemble.setMnemonic(KeyEvent.VK_L);
		JMenuItem file_close = new JMenuItem("Close");
		file_close.addActionListener(this);
		file_close.setMnemonic(KeyEvent.VK_C);
		
		file_menu.add(file_new);
		file_menu.add(file_open);
		file_menu.addSeparator();
		file_menu.add(file_save);
		file_menu.add(file_save_as);
		file_menu.addSeparator();
		file_menu.add(file_assemble);
		file_menu.add(file_close);
		
		menubar.add(file_menu);
		
		
		
		setJMenuBar(menubar);
		setContentPane(contentpane);
		addWindowListener(this);
		pack();
		setVisible(true);
		
	}
	
		
	
	private void newFile() {
		if (!saved) {
			int ret = JOptionPane.showConfirmDialog(this, "File not saved. Save now?");
			if (ret == JOptionPane.YES_OPTION) {
				saveFile();
			} else if (ret == JOptionPane.CANCEL_OPTION) {
				return;
			}			
		}
		
		filename = null;
		text.setText("");
		setSaved(true);
	}
	
	private void closeEditor() {
		if (!saved) {
			int ret = JOptionPane.showConfirmDialog(this, "File not saved. Save now?");
			if (ret == JOptionPane.YES_OPTION) {
				saveFile();
			} else if (ret == JOptionPane.CANCEL_OPTION) {
				this.setVisible(true);
				return;
			}
		}
		this.setVisible(false);
		this.dispose();
	}
	
	private void saveFile() {
		if (filename == null) {
			saveFileAs();
		} else {
			try {
				BufferedWriter bw = new BufferedWriter(new FileWriter(filename));
				bw.write(text.getText());
				bw.close();
				setSaved(true);
			} catch(IOException e) {
				JOptionPane.showMessageDialog(this, "An exception occured: " + e);
				e.printStackTrace();
			}
		}
	}
	
	private void saveFileAs() {
		JFileChooser fc = new JFileChooser(mf.getCurrentDirectory());
		fc.setFileFilter(new FileFilter() {
			
			public String getDescription() {
				return "Assembler code (*.asm)";
			}
		
			public boolean accept(File f) {
				if (f.getName().endsWith(".asm")) return true;
				if (f.isDirectory()) return true;
				return false;
			}
		
		});
		int ret = fc.showDialog(this, "Save");
		if (ret == JFileChooser.APPROVE_OPTION) {
			mf.setCurrentDirectory(fc.getCurrentDirectory());
			filename = fc.getSelectedFile().toString();
			saveFile();
		}
	}
	
	private void openFile() {
		if (!saved) {
			int ret = JOptionPane.showConfirmDialog(this, "File not saved. Save now?");
			if (ret == JOptionPane.YES_OPTION) {
				saveFile();
			} else if (ret == JOptionPane.CANCEL_OPTION) {
				return;
			}						
		}
		
		JFileChooser fc = new JFileChooser(mf.getCurrentDirectory());
		fc.setFileFilter(new FileFilter() {
		
			public String getDescription() {
				return "Assembler code (*.asm)";
			}
		
			public boolean accept(File f) {
				if (f.getName().endsWith(".asm")) return true;
				if (f.isDirectory()) return true;
				return false;
			}
		
		});
		int ret = fc.showDialog(this, "Open");
		if (ret == JFileChooser.APPROVE_OPTION) {
			mf.setCurrentDirectory(fc.getCurrentDirectory());
			filename = fc.getSelectedFile().toString();
			try {
				BufferedReader br = new BufferedReader(new FileReader(filename));
				String line;
				text.setText("");
				setSaved(true);
				while((line = br.readLine()) != null) {
					text.append(line);
					text.append("\n");
				}
				br.close();
                                text.setCaretPosition(0);
			} catch (IOException e) {
				JOptionPane.showMessageDialog(this, "An exception occured: " + e);
				e.printStackTrace();
			}
		}
	}
	
	private void setSaved(boolean b) {
		if (b) {
			if (filename == null) 
				this.setTitle(title);
			else
				this.setTitle(title + " (" + new File(filename).getName() + ")");
		} else {
			if (filename == null)
				this.setTitle(title + "*");
			else
				this.setTitle(title + "* (" + new File(filename).getName() + ")");
		}
		
		saved = b;
	}
	
	
	private void updateStatusbar() {
		int position = text.getCaretPosition();
		StringBuffer sb = new StringBuffer(text.getText());
		int line = 1;
		int column = 1;
		for(int i = 0; i < position; i++) {
			char c = sb.charAt(i);
			column++;
			if (c == '\n') {
				line++;
				column = 1;
			}
		}
		statusbar.setText("CaretPosition: " + text.getCaretPosition() + "  Line: " + line + "  Column: " + column);
	}
	
	
	
	
	public void actionPerformed(ActionEvent e) {
		String cmd = e.getActionCommand();
		if (cmd.equals("Assemble and load into simulator")) {
			mf.action_assemble(new StringBuffer(text.getText()));
		} else if (cmd.equals("Save")) {
			saveFile();
		} else if (cmd.equals("Save As")) {
			saveFileAs();
		} else if (cmd.equals("Close")) {
			closeEditor();
		} else if (cmd.equals("New")) {
			newFile();
		} else if (cmd.equals("Open")) {
			openFile();
		} else {
			System.out.println("Editor: ActionEvent!!");
		}
	}
	
	public void keyPressed(KeyEvent e) {
		if (e.isControlDown() && e.getKeyCode() == KeyEvent.VK_S) {
			saveFile();
		}
		
		if (!e.isControlDown() && !e.isAltDown()
				&& e.getKeyCode() != KeyEvent.VK_DOWN
				&& e.getKeyCode() != KeyEvent.VK_UP
				&& e.getKeyCode() != KeyEvent.VK_LEFT
				&& e.getKeyCode() != KeyEvent.VK_RIGHT) {
			setSaved(false);
		}
	}
	
	public void keyReleased(KeyEvent e) { }
	public void keyTyped(KeyEvent e) { }

	
	public void caretUpdate(CaretEvent e) {
		updateStatusbar();		
	}
	
	
	public void windowClosing(WindowEvent e) {
		closeEditor();
	}
	
	public void windowClosed(WindowEvent e) { } 
	public void windowActivated(WindowEvent e) { }
	public void windowDeactivated(WindowEvent e) { }
	public void windowDeiconified(WindowEvent e) { }
	public void windowIconified(WindowEvent e) { }
	public void windowOpened(WindowEvent e) { }
}
