import java.awt.BorderLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.filechooser.FileFilter;

public class MainFrame extends JFrame implements Runnable {
	private static final long serialVersionUID = 1234567890L;
		
	private JTextField pc = new JTextField(6);
	private JTextField ir = new JTextField(6);
	private JTextField ar = new JTextField(6);
	private JTextField accu = new JTextField(6);
	private JCheckBox zero = new JCheckBox("Zero-Flag");
	private JCheckBox carry = new JCheckBox("Carry-Flag");
	private JTextArea text = new JTextArea(22, 52);
	private EventListener listener = null;
	private JButton btn_run = new JButton("Run");
	
	private short[] ram = new short[4096];
	private Processor processor = new Processor(ram);
	
	private BreakpointsFrame breakpoints = new BreakpointsFrame();
	private Console console = new Console(ram);
	private LCD lcd = new LCD(ram);
	private RAMViewer ramviewer = new RAMViewer(ram, console, lcd, breakpoints);
	
	private File currentDirectory = new File(".");
	
	private Vector labels = new Vector();
	
	public MainFrame() {
		super("Toy Simulator");
		
		
		
		listener = new EventListener(this);
		pc.addActionListener(listener);
		ir.addActionListener(listener);
		ar.addActionListener(listener);
		accu.addActionListener(listener);
		zero.addActionListener(listener);
		carry.addActionListener(listener);

		
		JPanel content = new JPanel();
		content.setLayout(new BorderLayout());
		
		JPanel westpanel = new JPanel();
		westpanel.setLayout(new GridLayout(6,1));
		
		JPanel pc_panel = new JPanel();
		pc_panel.add(new JLabel("PC"));
		pc_panel.add(pc);
		
		JPanel ir_panel = new JPanel();
		ir_panel.add(new JLabel("IR"));
		ir_panel.add(ir);
		
		JPanel ar_panel = new JPanel();
		ar_panel.add(new JLabel("AR"));
		ar_panel.add(ar);
		
		JPanel accu_panel = new JPanel();
		accu_panel.add(new JLabel("Accu"));
		accu_panel.add(accu);
		
		JPanel buttons_panel = new JPanel();
		buttons_panel.setLayout(new GridLayout(1,4));
		
		JButton btn_reset = new JButton("Reset");
		btn_reset.addActionListener(listener);
		
		JButton btn_singlestep = new JButton("Single Step");
		btn_singlestep.addActionListener(listener);
		
		btn_run.addActionListener(listener);
		
		JButton btn_breakpoint = new JButton("Breakpoints");
		btn_breakpoint.addActionListener(listener);
		
		buttons_panel.add(btn_breakpoint);
		buttons_panel.add(btn_singlestep);
		buttons_panel.add(btn_run);
		buttons_panel.add(btn_reset);
		
		
		westpanel.add(pc_panel);
		westpanel.add(ir_panel);
		westpanel.add(ar_panel);
		westpanel.add(accu_panel);
		westpanel.add(zero);
		westpanel.add(carry);
		
		content.add(westpanel, BorderLayout.WEST);
		content.add(buttons_panel, BorderLayout.SOUTH);
		text.setFont(new Font("monospaced", 0, 10));
		content.add(new JScrollPane(text), BorderLayout.CENTER);
		
		
		JMenuBar menubar = new JMenuBar();
		JMenu file = new JMenu("File");
		file.setMnemonic(KeyEvent.VK_F);
		JMenuItem file_exit = new JMenuItem("Exit");
		file_exit.setMnemonic(KeyEvent.VK_X);
		JMenuItem file_open_object_file = new JMenuItem("Open Object File...");
		file_open_object_file.setMnemonic(KeyEvent.VK_O);
		JMenuItem file_open_asm_file = new JMenuItem("Open Assembler File...");
		file_open_asm_file.setMnemonic(KeyEvent.VK_A);
		JMenuItem file_new_asm = new JMenuItem("New Assembler File...");
		file_new_asm.setMnemonic(KeyEvent.VK_N);
		
		file_exit.addActionListener(listener);
		file_open_object_file.addActionListener(listener);
		file_open_asm_file.addActionListener(listener);
		file_new_asm.addActionListener(listener);
		
		file.add(file_new_asm);
		file.addSeparator();
		file.add(file_open_object_file);
		file.add(file_open_asm_file);
		file.addSeparator();
		file.add(file_exit);
		
		JMenu view = new JMenu("View");
		view.setMnemonic(KeyEvent.VK_V);
		JMenuItem view_ram = new JMenuItem("RAM");
		view_ram.setMnemonic(KeyEvent.VK_R);
		JMenuItem view_console = new JMenuItem("Console");
		view_console.setMnemonic(KeyEvent.VK_C);
		JMenuItem view_lcd = new JMenuItem("LCD");
		view_lcd.setMnemonic(KeyEvent.VK_L);
		
		view_ram.addActionListener(listener);
		view_console.addActionListener(listener);
		view_lcd.addActionListener(listener);
		
		view.add(view_ram);
		view.add(view_console);
		view.add(view_lcd);
		
		JMenu about = new JMenu("About");
		about.setMnemonic(KeyEvent.VK_A);
		JMenuItem about_version = new JMenuItem("Version");
		about_version.setMnemonic(KeyEvent.VK_V);
		
		about_version.addActionListener(listener);
		
		about.add(about_version);
		
		menubar.add(file);
		menubar.add(view);
		menubar.add(about);
		
		
		setJMenuBar(menubar);
		setContentPane(content);
		
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);		
		
		pack();
		setVisible(true);
		
		
		
		
	    
		program();
		action_reset();
		showProcessor();
	}
	
	private String toHexString(int i, int bits, boolean prefix) {
		if (i < 0) i += Math.pow(2, bits);
		String s = Integer.toHexString(i);
		while (s.length() < (bits/4)) {
			s = "0" + s;
		}
		
		if (prefix) {
			return "0x" + s;
		} else {
			return s;
		}
	}
	
	private void showProcessor() {
		pc.setText(toHexString(processor.getPC(), 12, true));
		ir.setText(toHexString(processor.getIR(), 16, true));
		ar.setText(toHexString(processor.getAR(), 12, true));
		accu.setText(toHexString(processor.getAccu(), 16, true));
		zero.setSelected(processor.getZeroFlag());
		carry.setSelected(processor.getCarryFlag());
	}
	
	public void action_new_assembler_file() {
		new Editor(this);
	}
	
	public void action_breakpoints() {
		breakpoints.setVisible(true);
	}
	
	public void action_reset() {
		text.setText("");
		processor.reset();
		ramviewer.loadRAM();
		ramviewer.markRow(processor.getPC());
		console.showText();
		lcd.showText();
		showProcessor();
	}
	
	public void action_singlestep() {
		text.append(disassembleIR());
		text.setCaretPosition(text.getText().length());
		processor.single_step();
		ramviewer.loadRAM();
		ramviewer.markRow(processor.getPC());
		console.showText();
		lcd.showText();
		showProcessor();
	}
	
	public void action_run() {
		new Thread(this).start();
	}
	
	public void action_view_ram() {
		ramviewer.setVisible(true);
	}
	
	public void action_view_console() {
		console.setVisible(true);
	}
	
	public void action_view_lcd() {
		lcd.setVisible(true);
	}
	
	public void action_about_version() {
		JOptionPane.showMessageDialog(this, Main.VERSION_STRING);
	}
	
	public void run() {
		while(true) {
			text.append(disassembleIR());
			text.setCaretPosition(text.getText().length());
			processor.single_step();
			ramviewer.markRow(processor.getPC());
			console.showText();
			lcd.showText();
			showProcessor();
			if (breakpoints.isBreakpoint(processor.getPC())) {
				btn_run.setText("Run");
				break;
			}

			Thread.yield();
			
			if (btn_run.getText().equals("Run")) {
				break;
			}
		}
		ramviewer.loadRAM();
		showProcessor();
	}
	
	
	private void program() {
		
		ram[0x0] = 0x000C; // STO 0x00C
		ram[0x1] = 0x400C; // SUB 0x00C
		ram[0x2] = 0x100D; // LDA 0x00D
		ram[0x3] = 0x300E; // ADD 0x00E
		ram[0x4] = 0x100B; // LDA 0x00B
		ram[0x5] = 0x2010; // BRZ 0x010
		
		ram[0xB] = 0x0000;
		ram[0xC] = 0x0000;
		ram[0xD] = 0x0001;
		ram[0xE] = 0x0E20;
		ram[0xF] = 0x5210;
		ram[0x10] = 0x300F;	// ADD 0x00F
		
		
		/* test carry flag: SHL
		ram[0x0] = 0x1002; // LDA 0x002
		ram[0x1] = (short) 0xD000; // SHL
		ram[0x2] = (short) 0x8000; // 0x8000
		*/
		
		/* test carry flag: ADD
		ram[0x0] = 0x1003; // LDA 0x003
		ram[0x1] = 0x3004; // ADD 0x004
		ram[0x2] = 0x3004; // ADD 0x004
		ram[0x3] = 0x0;
		ram[0x4] = (short) 0xfff0;
		*/
		
		/* test carry flag: SUB
		ram[0x0] = 0x1003; // LDA 0x003
		ram[0x1] = 0x4004; // SUB 0x004
		ram[0x2] = 0x4004; // SUB 0x004
		ram[0x3] = 0x0;
		ram[0x4] = (short) 0xfff0;
		*/
		
		ram[0xf38] = (('h' & 0xFF) << 8) + ('e' & 0xFF);
		ram[0xf39] = 0;
		ram[0xf3a] = 'l';
		ram[0xf3b] = 'l';
		ram[0xf3c] = 'o';
		ram[0xf3d] = ' ';
		ram[0xf3e] = 'w';
		ram[0xf3f] = 'o';
		ram[0xf40] = 'r';
		ram[0xf41] = 'l';
		ram[0xf42] = 'd';
		ram[0xf43] = '!';
		
		ram[0xffe] = (('A' & 0xFF) << 8) | ('B' & 0xFF);
		ram[0xfff] = (('C' & 0xFF) << 8) | ('D' & 0xFF);
		
		ramviewer.loadRAM();
		console.showText();
		lcd.showText();
	}
		
	public void action_open_object_file() {
		JFileChooser fc = new JFileChooser(currentDirectory);
		fc.setFileFilter(new FileFilter() {
		
			public String getDescription() {
				return "Object files (*.o; *.obj)";
			}
		
			public boolean accept(File f) {
				if (f.getName().endsWith(".o")
						|| f.getName().endsWith(".obj")) {
					return true;
				} else {
					if (f.isDirectory()) return true;
					return false;
				}
			}
		
		});
		
		if (fc.showDialog(this, "Open") == JFileChooser.APPROVE_OPTION) {
			currentDirectory = fc.getCurrentDirectory();
			try {
				FileInputStream in = new FileInputStream(fc.getSelectedFile());
				int c;
				short s = 0;
				short index = 0;
				while((c = in.read()) != -1) {
					s += ((c & 0xFF) << 8);
					c = in.read();
					s += c & 0xFF;
					ram[index++] = s;
					s = 0;
				}
				in.close();
			} catch (IOException e) {
				e.printStackTrace();
				JOptionPane.showMessageDialog(this, "Exception occured: " + e);
			}
			
			action_reset();
			showProcessor();
		}
		
	}
	
	public void action_open_asm_file() {
		JFileChooser fc = new JFileChooser(currentDirectory);
		fc.setFileFilter(new FileFilter() {
		
			public String getDescription() {
				return "Assembler code (*.asm)";
			}
		
			public boolean accept(File f) {
				if (f.getName().endsWith(".asm")) return true;
				if (f.isDirectory()) return true;
				return false;
			}
		
		});
		if (fc.showDialog(this, "Open") == JFileChooser.APPROVE_OPTION) {
			currentDirectory = fc.getCurrentDirectory();
			try {
				Assembler.assemble(fc.getSelectedFile(), ram, labels);
				ramviewer.setLabels(labels);
			} catch (Assembler.ParserException pe) {
				pe.printStackTrace();
				JOptionPane.showMessageDialog(this, "ParserException: " + pe.getMessage());
			} catch (IOException e) {
				e.printStackTrace();
				JOptionPane.showMessageDialog(this, "Exception ocurred: " + e);
			}
			
			action_reset();
			showProcessor();
		}
	}
	
	public void action_assemble(StringBuffer sb) {
		try  {
			Assembler.assemble(sb, ram, labels);
			ramviewer.setLabels(labels);
		} catch (Assembler.ParserException pe) {
			pe.printStackTrace();
			JOptionPane.showMessageDialog(this, "ParserException: " + pe.getMessage());
		}
		
		action_reset();
		showProcessor();
	}
	
	private String disassembleIR() {
		short ir = processor.getIR();
		byte opcode = (byte) ((ir & 0xF000) >> 12);
		short address = (short) (ir & 0x0FFF);
		String s_address = toHexString(address, 12, true);
		
		String s = "";
		
		switch (opcode) {
			case Processor.ADD:
				s += "ADD " + s_address + "  ";
				s += "mem[" + s_address + "] = " + toHexString(ram[address], 16, true) + "\n";
				break;				
			case Processor.AND:
				s += "AND " + s_address + "  ";
				s += "mem[" + s_address + "] = " + toHexString(ram[address], 16, true) + "\n";
				break;
			case Processor.BRC:
				s += "BRC " + s_address + "\n";
				break;
			case Processor.BRZ:
				s += "BRZ " + s_address + "\n";
				break;
			case Processor.CPI:
				s += "CPI\n";
				break;
			case Processor.INCAR:
				s += "INCAR\n";
				break;
			case Processor.LDA:
				s += "LDA " + s_address + "  ";
				s += "mem[" + s_address + "] = " + toHexString(ram[address], 16, true) + "\n";
				break;
			case Processor.LDAR:
				s += "LDAR " + s_address + "\n";
				break;
			case Processor.LDI:
				s += "LDI\n";
				break;
			case Processor.OR:
				s += "OR " + s_address + "  ";
				s += "mem[" + s_address + "] = " + toHexString(ram[address], 16, true) + "\n";
				break;
			case Processor.SHL:
				s += "SHL\n";
				break;
			case Processor.SHR:
				s += "SHR\n";
				break;
			case Processor.STI:
				s += "STI\n";
				break;
			case Processor.STO:
				s += "STO " + toHexString(address, 12, true) + "\n";
				break;
			case Processor.SUB:
				s += "SUB " + s_address + "  ";
				s += "mem[" + s_address + "] = " + toHexString(ram[address], 16, true) + "\n";
				break;
			case Processor.XOR:
				s += "XOR " + s_address + "  ";
				s += "mem[" + s_address + "] = " + toHexString(ram[address], 16, true) + "\n";
				break;
				
		}
		
		return s;
	}
	
	public void setFlags() {
		processor.setZeroFlag(zero.isSelected());
		processor.setCarryFlag(carry.isSelected());
	}
	
	public void setRegister(JTextField f) {
		try {
			short s;
			if (f.getText().startsWith("0x")) {
				s = (short) Integer.parseInt(f.getText().substring(2), 16);
			} else {
				s = (short) Integer.parseInt(f.getText());
			}

			if (f == pc) {
				processor.setPC(s);
			} else if (f == ir) {
				processor.setIR(s);
			} else if (f == ar) {
				processor.setAR(s);
			} else if (f == accu) {
				processor.setAccu(s);
			}
		} catch (NumberFormatException e) {
			JOptionPane.showMessageDialog(this, "Error: Couldn't convert number: " + f.getText());
		}
		showProcessor();
	}
	
	public File getCurrentDirectory() {
		return currentDirectory;
	}
	
	public void setCurrentDirectory(File f) {
		currentDirectory = f;
	}
}
