
public class Processor {
	private short[] mem;	// RAM
	private short pc;		// Program Counter
	private short ir;		// Intruction Register
	private short ar;		// Address Register
	private short accu;		// Accumulator
	private boolean zero;	// Zero Flag
	private boolean carry;  // Carry Flag
	
	
	public static final byte STO = 0x0;
	public static final byte LDA = 0x1;
	public static final byte BRZ = 0x2;
	public static final byte ADD = 0x3;
	public static final byte SUB = 0x4;
	public static final byte OR  = 0x5;
	public static final byte AND = 0x6;
	public static final byte XOR = 0x7;
	public static final byte LDAR = 0x8;
	public static final byte INCAR = 0x9;
	public static final byte LDI = 0xA;
	public static final byte STI = 0xB;
	public static final byte CPI = 0xC;
	public static final byte SHL = 0xD;
	public static final byte SHR = 0xE;
	public static final byte BRC = 0xF;
	
	
	
	public Processor(short[] m) {
		mem = m;
	}
	
	public void reset() {
		pc = 0;
		// fetch next instruction
		ir = mem[pc];
	}
	
	public void single_step() {
		byte opcode;
		short address;
		
		
		// decode the instruction in the ir
		
		// extract opcode (4 bits)
		opcode = (byte)((ir & 0xF000) >> 12);
		// extract address (12 bits)
		address = (short) (ir & 0x0FFF);

		
		// increment program counter
		pc++;
		// wrap around program counter
		if (pc >= 4096) pc = 0;
		
				

		switch (opcode) {
		
			case STO:
				mem[address] = accu;
				zero = (accu == 0);
				carry = false;
				break;
				
			case LDA:
				accu = mem[address];
				zero = (accu == 0);
				carry = false;
				break;
				
			case BRZ:
				if (zero) pc = address;
				zero = (accu == 0);
				carry = false;
				break;
				
			case ADD:
				if (((accu&0xffff) + (mem[address]&0xffff) & 0x10000) == 0x10000) {
					carry = true;
				}
				accu += mem[address];
				zero = (accu == 0);
				break;
				
			case SUB:
				//if (((accu&0xffff) - (mem[address]&0xffff) & 0x10000) == 0x10000) {
				//	carry = true;
				//}
				if ((accu&0xffff) < (mem[address]&0xffff)) {
					carry = true;
				}
				accu -= mem[address];
				zero = (accu == 0);
				break;
				
			case OR:
				accu = (short) (accu | mem[address]);
				zero = (accu == 0);
				carry = false;
				break;
				
			case AND:
				accu = (short) (accu & mem[address]);
				zero = (accu == 0);
				carry = false;
				break;
				
			case XOR:
				accu = (short) (accu ^ mem[address]);
				zero = (accu == 0);
				carry = false;
				break;
				
			case LDAR:
				ar = address;
				zero = (accu == 0);
				carry = false;
				break;
				
			case INCAR:
				ar++;
				if (ar > 0x1000) ar = 0;
				zero = (accu == 0);
				carry = false;
				break;
				
			case LDI:
				accu = mem[ar];
				zero = (accu == 0);
				carry = false;
				break;
				
			case STI:
				mem[ar] = accu;
				zero = (accu == 0);
				carry = false;
				break;
				
			case CPI:
				zero = (accu == mem[ar]);
				carry = false;
				break;
				
			case SHL:
				//if ((((accu&0xffff) << 1) & 0x10000) == 0x10000) {
				//	carry = true;
				//}
				if ((accu & 0x8000) == 0x8000) {
					carry = true;
				}
				accu = (short) (accu << 1);
				zero = (accu == 0);
				break;
				
			case SHR:
				if ((accu & 0x1) == 0x1) {
					carry = true;
				}
				accu = (short) (accu >> 1);
				zero = (accu == 0);
				break;
				
			case BRC:
				if (carry) pc = address;
				zero = (accu == 0);
				carry = false;
				break;
		
		}


		// read the next instruction and increment pc
		ir = mem[pc];
		
			
	}
	
	
	public short[] getMem() { return mem; }
	public short getPC() { return pc; }
	public short getIR() { return ir; }
	public short getAR() { return ar; }
	public short getAccu() { return accu; }
	public boolean getZeroFlag() { return zero; }
	public boolean getCarryFlag() { return carry; }
		
	
	public void setPC(short s) { pc = s; }
	public void setIR(short s) { ir = s; }
	public void setAR(short s) { ar = s; }
	public void setAccu(short s) { accu = s; zero = (accu == 0); }
	public void setZeroFlag(boolean b) { zero = b; }
	public void setCarryFlag(boolean b) { carry = b; }
}
