import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Vector;

import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

public class RAMViewer extends JFrame implements TableModelListener, MouseListener {	
	private static final long serialVersionUID = 1234567895L;

	private JTable table;
	private short[] mem;
	private boolean listener_off = false;
	private Console console;
	private LCD lcd;
	private BreakpointsFrame breakpoints;

	public RAMViewer(short[] mem, Console c, LCD l, BreakpointsFrame breakpoints) {
		super("Toy Simulator - RAM Viewer");
		
		this.mem = mem;
		console = c;
		lcd = l;
		this.breakpoints = breakpoints;
		
		table = new JTable(mem.length, 5);
		
		table.getColumnModel().getColumn(0).setHeaderValue("Breakpoint");
		table.getColumnModel().getColumn(1).setHeaderValue("Address");
		table.getColumnModel().getColumn(2).setHeaderValue("Value");
		table.getColumnModel().getColumn(3).setHeaderValue("Code");
		table.getColumnModel().getColumn(4).setHeaderValue("Label");
		
		loadRAM();
		
		table.getModel().addTableModelListener(this);
		table.addMouseListener(this);
		
		getContentPane().add(new JScrollPane(table));
		
		pack();
		setVisible(false);
	}
	
	public void tableChanged(TableModelEvent e) {
		int row = e.getFirstRow();
		int col = e.getColumn();
		
		if (listener_off) return;
		
		if (col == 1) {
			try {
				short s;
				String value = (String) table.getModel().getValueAt(row, col);
				if (value.startsWith("0x") || value.startsWith("0X")) {
					s = (short) Integer.parseInt(value.substring(2), 16);
				} else if (value.startsWith("$")) {
					s = (short) Integer.parseInt(value.substring(1), 16);
				} else if (value.startsWith("0")) {
					s = (short) Integer.parseInt(value, 7);
				} else if (value.endsWith("b") || value.endsWith("B")) {
					s = (short) Integer.parseInt(value.substring(0, value.length()-1), 2);
				} else {
					s = (short) Integer.parseInt(value);
				}

				mem[row] = s;
				console.showText();
				lcd.showText();
			} catch (NumberFormatException ex) {
				ex.printStackTrace();
			}
		}
		
		updateRow(row);		
	}
	
	public void updateRow(int row) {
		listener_off = true;
		table.getModel().setValueAt(toHexString(row, 12, true), row, 1);
		table.getModel().setValueAt(toHexString(mem[row], 16, true), row, 2);
		table.getModel().setValueAt(disassemble(mem[row]), row, 3);
		listener_off = false;
	}
	
	private String disassemble(short ir) {
		byte opcode = (byte) ((ir & 0xF000) >> 12);
		short address = (short) (ir & 0xFFF);

		String s = "";
		String s_address = toHexString(address, 12, true);
		
		switch (opcode) {
		case Processor.ADD:
			s += "ADD " + s_address;
			break;				
		case Processor.AND:
			s += "AND " + s_address;
			break;
		case Processor.BRC:
			s += "BRC " + s_address;
			break;
		case Processor.BRZ:
			s += "BRZ " + s_address;
			break;
		case Processor.CPI:
			s += "CPI";
			break;
		case Processor.INCAR:
			s += "INCAR";
			break;
		case Processor.LDA:
			s += "LDA " + s_address;
			break;
		case Processor.LDAR:
			s += "LDAR " + s_address;
			break;
		case Processor.LDI:
			s += "LDI";
			break;
		case Processor.OR:
			s += "OR " + s_address;
			break;
		case Processor.SHL:
			s += "SHL";
			break;
		case Processor.SHR:
			s += "SHR";
			break;
		case Processor.STI:
			s += "STI";
			break;
		case Processor.STO:
			s += "STO " + s_address;
			break;
		case Processor.SUB:
			s += "SUB " + s_address;
			break;
		case Processor.XOR:
			s += "XOR " + s_address;
			break;
		}

		
		return s;
	}
	
	private String toHexString(int i, int bits, boolean prefix) {
		if (i < 0) i += Math.pow(2, bits);
		String s = Integer.toHexString(i);
		while (s.length() < (bits/4)) {
			s = "0" + s;
		}
		
		if (prefix) {
			return "$" + s;
		} else {
			return s;
		}
	}
	
	public void loadRAM() {
		listener_off = true;
		for(int i = 0; i < mem.length; i++) {
			table.getModel().setValueAt(toHexString(i, 12, true), i, 1);
			table.getModel().setValueAt(toHexString(mem[i], 16, true), i, 2);
			table.getModel().setValueAt(disassemble(mem[i]), i, 3);
		}
		listener_off = false;
	}
		
	public void markRow(int row) {
		table.changeSelection(row, 0, false, false);
	}
	
	public void setLabels(Vector labels) {
		if (labels == null) return;
		
		for(int i = 0; i < mem.length; i++) {
			table.getModel().setValueAt("", i, 4);
		}
		
		for(int i = 0; i < labels.size(); i++) {
			Assembler.Identifier l = (Assembler.Identifier) labels.get(i);
			String newValue = l.getIdentifier();
			if (l.getType() == Assembler.Identifier.TYPE_LABEL) {
				newValue += ":";
			}
			String oldValue = (String) table.getModel().getValueAt(l.getIndex(), 4);
			if (oldValue.length() > 0) {
				oldValue = oldValue + ", ";
			}
			table.getModel().setValueAt(oldValue + newValue, l.getIndex(), 4);
		}
	}

	public void updateBreakpoints() {
		for(int i = 0; i < mem.length; i++) {
			if (breakpoints.isBreakpoint(i)) {
				table.getModel().setValueAt("Y", i, 0);
			} else {
				table.getModel().setValueAt("", i, 0);
			}
		}
	}
	
	
	public void mouseClicked(MouseEvent e) { }
	public void mouseEntered(MouseEvent e) { }
	public void mouseExited(MouseEvent e) { }
	public void mousePressed(MouseEvent e) {
		if (e.getButton() == MouseEvent.BUTTON3) {
			
			int row = table.rowAtPoint(e.getPoint());
			
			if (row > -1) {
				if (breakpoints.isBreakpoint(row)) {
					breakpoints.removeBreakpoint((String) table.getModel().getValueAt(row, 1));
				} else {
					breakpoints.addBreakpoint((String) table.getModel().getValueAt(row, 1));
				}
				updateBreakpoints();
			}
		}

	}
	public void mouseReleased(MouseEvent e) { }
}
