/*
 *  src/vocable.cpp
 *  2002-05-27
 * 
 *  voctrain - a vocabulary trainer for linux and windows
 *  Copyright (C) 2002  Andreas Dangel <a.dangel@gmx.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "vocable.h"
#include <qstring.h>
#include <qstringlist.h>
#include <qlistview.h>


/**
 * eine (static) Variable zur Umwandlung von Wortart im int-Format (Type)
 * in eine Zeichenkette.
 */
char * Vocable::TYPES[6] = { "noun", "verb", "adjective", "pronoun",
                             "invariable", "other" };


/**
 * Konstruktor: Initialisiert alle Felder.
 * Leert alle QString-Objekte mit "" und aType mit eType_Other.
 */
Vocable::Vocable(Lesson * parent) : QListViewItem(parent)
{
    aWord = "";
    aTranslations.clear();
    aRemark = "";
    aTag = "";
    aType = eType_Other;
    toXML();
}

/**
 * Destruktor: keine Funktion/Aufgabe
 */
Vocable::~Vocable() { }

/**
 * Verndert das Attribut aWord.
 */
void Vocable::setWord(const QString& s)
{
    aWord = s.latin1();
}

/**
 * Verndert das Attribut aRemark.
 */
void Vocable::setRemark(const QString& s)
{
    aRemark = s.latin1();
}

/**
 * Verndert das Attribut aTag.
 */
void Vocable::setTag(const QString& s)
{
    aTag = s.latin1();
}

/**
 * Verndert das Attribut aType.
 */
void Vocable::setType(const Type t)
{
    aType = t;
}



/**
 * Gibt aWord zurck.
 */
const QString& Vocable::getWord()
{
    return aWord;
}

/**
 * Gibt aRemark zurck.
 */
const QString& Vocable::getRemark()
{
    return aRemark;
}

/**
 * Gibt aTag zurck.
 */
const QString& Vocable::getTag()
{
    return aTag;
}

/**
 * Gibt aType zurck.
 */
const Vocable::Type Vocable::getType()
{
    return aType;
}


/**
 * Gibt aTranslations zurck.
 * Der Rckgabewert ist eine Referenz und vernderbar, d.h.
 * ber diese Referenz lassen sich die Translations verndern.
 */
QStringList& Vocable::getTranslations()
{
    return aTranslations;
}


/**
 * Erstellt einen XML-Code.
 */
const QString& Vocable::toXML(int indent = 0)
{
    /* Zeileneinzug */
    QString sIndent = "";
    for (int i = 0; i < indent; i++) sIndent += " ";

    aXML  = "";
    aXML  = sIndent + "<Vocable>\n";
    aXML += sIndent + "  <Word type=\"" + TYPES[aType] + "\"";
    if (aTag.length() > 0)
    {
        aXML += " tag=\"" + aTag + "\"";
    }
    aXML += ">" + aWord + "</Word>\n";
    for (uint i = 0; i < aTranslations.count(); i++)
    {
        aXML += sIndent + "  <Translation>" + aTranslations[i];
        aXML += "</Translation>\n";
    }
    if (aRemark.length() > 0)
    {
        aXML += sIndent + "  <Remark>" + aRemark + "</Remark>\n";
    }
    aXML += sIndent + "</Vocable>\n";

    return aXML;
}


/**
 * berschriebene Methode von QListViewItem
 */
QString Vocable::text(int column) const
{
    QString t;
    switch(column) {
        case 0: return "VOCABLE";
        case 1: return aWord;
        case 2:
            t = "";
            for (uint i = 0; i < aTranslations.count(); i++)
            {
                t += aTranslations[i];
                if (i < aTranslations.count()-1) t += ", ";
            }
            return t;
        case 3: return aRemark;
        default: return "";
    }
}


/**
 * berschriebene Methoden von Item
 */
bool Vocable::isVocable() { return true; }
bool Vocable::isLesson() { return false; }
