/*
 *  vocable.h
 *  2002-05-27
 *
 *  voctrain - a vocabulary trainer for linux and windows
 *  Copyright (C) 2002  Andreas Dangel <a.dangel@gmx.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _VOCABLE_H_
#define _VOCABLE_H_

#include "lesson.h"
#include "item.h"

// verwendet die qt-Library 2.3 von trolltech.com
#include <qstring.h>
#include <qstringlist.h>
#include <qlistview.h>


/**
 * Klasse     : Vocable
 * Summary    : Reprsentiert den Datensatz einer Vokabel.
 * Description: Die Klasse Vocable wird von QListViewItem abgeleitet, sodass
 *     sie leicht in einem QListView-Objekt verwendet werden kann. Zum
 *     Datensatz gehren das Wort, die Wortart, die bersetzungen,
 *     eine Bemerkung und ein zustzliches Feld ("tag").
 */
class Vocable : public QListViewItem, public Item
{
    public:
        /**
         * Konstruktor: erstellt ein leeres Vocable-Object mit leeren Feldern.
         *     Das parent-Argument wird an den geerbten Konstruktor bergeben.
         */
        Vocable(Lesson * parent);

        /**
         * Destruktor: keine Funktion/Aufgabe
         */
        ~Vocable();

        /**
         * Diese Aufzhlung definiert die verschiedenen Wortarten.
         */
        enum Type {
            eType_Noun = 0,
            eType_Verb,
            eType_Adjective,
            eType_Pronoun,
            eType_Invariable,
            eType_Other
        };

        /**
         * Diese static Variable erleichtert die Umwandlung von Wortart-Typ
         * in eine Zeichenkette.
         */
        static char * TYPES[6];

        
        /**
         * Verndert das Attribut aWord.
         */
        void setWord(const QString& s);

        /**
         * Verndert das Attribut aRemark.
         */
        void setRemark(const QString& s);

        /**
         * Verndert das Attribut aTag.
         */
        void setTag(const QString& s);

        /**
         * Verndert das Attribut aType.
         */
        void setType(const Type t);


        /**
         * Gibt aWord zurck.
         */
        const QString& getWord();

        /**
         * Gibt aRemark zurck.
         */
        const QString& getRemark();

        /**
         * Gibt aTag zurck.
         */
        const QString& getTag();

        /**
         * Gibt aType zurck.
         */
        const Type getType();


        /**
         * Gibt aTranslations zurck.
         * Der Rckgabewert ist eine Referenz und vernderbar, d.h.
         * ber diese Referenz lassen sich die Translations verndern.
         */
        QStringList& getTranslations();


        /**
         * Erstellt einen XML-Code.
         */
        const QString& toXML(int indent = 0);


        /**
         * berschriebene Methode von QListViewItem
         */
        QString text(int column) const;

        /**
         * berschriebene Methoden von Item
         */
        bool isVocable();
        bool isLesson();
        bool isVocabulary();

    private:
        /** Wort */
        QString aWord;
        /** Liste mit bersetzungen */
        QStringList aTranslations;
        /** Erklrung bzw. Bemerkung */
        QString aRemark;
        /** Wortart */
        Type aType;
        /** Tag */
        QString aTag;

        /** XML */
        QString aXML;
};

#endif _VOCABLE_H_
