/*
 *  src/vocabulary.cpp
 *  2002-05-27
 *
 *  voctrain - a vocabulary trainer for linux and windows
 *  Copyright (C) 2002  Andreas Dangel <a.dangel@gmx.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "vocabularyparser.h"
#include "lesson.h"
#include <qlistview.h>
#include <qstring.h>
#include <qwidget.h>
#include <qxml.h>
#include <qfile.h>
#include <qtextstream.h>

/**
 * Konstruktor: Initialisert alle Felder.
 */
Vocabulary::Vocabulary(QWidget * parent = 0, const char * name = 0) : QListView(parent, name)
{
    aTitle = "";
    aDescription = "";
    aDate = "";
    aAuthor = "";
    toXML();

    this->addColumn("type");
    this->addColumn("word / title");
    this->addColumn("translations");
    this->addColumn("remark");
    this->setRootIsDecorated(true);
    this->setShowSortIndicator(true);
    this->setSorting(1, true); //Column "word / title"
    this->setAllColumnsShowFocus(true);
}

/**
 * Destruktor: keine Funktion/Aufgabe
 */
Vocabulary::~Vocabulary() { }


/**
 * Verndert das Attribut aTitle
 */
void Vocabulary::setTitle(const QString& s)
{
    aTitle = s.latin1();
}

/**
 * Verndert das Attribut aDescription
 */
void Vocabulary::setDescription(const QString& s)
{
    aDescription = s.latin1();
}

/**
 * Verndert das Attribut aDate.
 */
void Vocabulary::setDate(const QString& s)
{
    aDate = s.latin1();
}

/**
 * Verndert das Attribut aAuthor.
 */
void Vocabulary::setAuthor(const QString& s)
{
    aAuthor = s.latin1();
}


/**
 * Gibt aTitle zurck.
 */
const QString& Vocabulary::getTitle()
{
    return aTitle;
}

/**
 * Gibt aDescription zurck.
 */
const QString& Vocabulary::getDescription()
{
    return aDescription;
}

/**
 * Gibt aDate zurck.
 */
const QString& Vocabulary::getDate()
{
    return aDate;
}

/**
 * Gibt aAuthor zurck.
 */
const QString& Vocabulary::getAuthor()
{
    return aAuthor;
}


/**
 * Erstellt einen XML-Code.
 */
const QString& Vocabulary::toXML(int indent = 0)
{
    /* Zeileneinzug */
    QString sIndent = "";
    for (int i = 0; i < indent; i++) sIndent += " ";

    aXML  = "";
    aXML += "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n";
    aXML += "<!DOCTYPE Vocabulary SYSTEM \"vocabulary.dtd\">\n";
    aXML += "\n";

    aXML += sIndent + "<Vocabulary>\n";
    aXML += sIndent + "  <TitleOfVocabulary>" + aTitle + "</TitleOfVocabulary>\n";
    
    if (aDescription.length() > 0) {
        aXML += sIndent + "  <Description>" + aDescription + "</Description>\n";
    }
    if (aDate.length() > 0) {
        aXML += sIndent + "  <Date>" + aDate + "</Date>\n";
    }
    if (aAuthor.length() > 0) {
        aXML += sIndent + "  <Author>" + aAuthor + "</Author>\n";
    }
    aXML += "\n";

    /* Lektionen */
    QListViewItem * it = this->firstChild();

    while (it != NULL)
    {
        Item * it2 = dynamic_cast<Item*> (it);
        if (it2->isLesson() == true) {
            aXML += ((Lesson *) it)->toXML(indent + 2);
        }

        it = it->nextSibling();
    }

    aXML += sIndent + "</Vocabulary>\n";

    return aXML;
}

/**
 * Ldt einen XML-Code und erstellt entsprechende Vocable- und
 * Lesson-Objekte.
 */
void Vocabulary::parseXML(const QString& s)
{
    VocabularyParser handler (this);
    QXmlInputSource source;
    source.setData(s);
    QXmlSimpleReader reader;
    reader.setContentHandler(&handler);
    reader.parse(source);
}

/**
 * ffnet eine Datei und parst deren Inhalt.
 */
void Vocabulary::loadFromFile(const QString& s)
{
    QFile f (s);
    if (!f.open(IO_ReadOnly)) return;
    QTextStream t(&f);
    QString source = "";
    while (!t.eof())
    {
        source += t.readLine();
    }
    f.close();

    this->parseXML(source);
}

/**
 * Speichert die Reprsentierte Lektionsdatei in eine Datei
 * im XML-Format.
 */
void Vocabulary::saveToFile(const QString& s)
{
    QFile f (s);
    if (!f.open(IO_WriteOnly)) return;
    QTextStream t(&f);
    t << this->toXML(0);
    f.flush();
    f.close();
}
